--[[
Interface: 1.4.1.0 b5334

Copyright (C) GtX (Andy), 2018

Author: GtX | Andy
Date: 15.03.2018
Version: 1.0.0.0

Contact:
https://forum.giants-software.com
https://github.com/GtX-Andy

History:
V 1.0.0.0 @ 15.03.2018 - FS17 Release
V 1.1.0.0 @ 28.08.2019 - FS19 Release

Important:
No changes are to be made to this script without permission from GtX | Andy
An diesem Skript dürfen ohne Genehmigung von GtX | Andy keine Änderungen vorgenommen werden
]]


HandPlantSaplingsShovel = {}
local HandPlantSaplingsShovel_mt = Class(HandPlantSaplingsShovel, HandTool)

InitObjectClass(HandPlantSaplingsShovel, "HandPlantSaplingsShovel")

HandPlantSaplingsShovel.INDEX_TO_COLOUR = {
    {0.8069, 0.0097, 0.0097, 1},
    {0.395, 0.925, 0.115, 1},
    {0.098, 0.450, 0.960, 1}
}

function HandPlantSaplingsShovel:new(isServer, isClient, customMt)
    local self = HandPlantSaplingsShovel:superClass().new(self, isServer, isClient, customMt or HandPlantSaplingsShovel_mt)

    self.i18n = g_i18n

    self.isShovel = true
    self.storage = nil

    self.inputWaitTimer = 0

    self.placeHolderActive = false
    self.placeHolderToggle = true

    self.nodeLocation = {x = 0, y = 0, z = 0}

    self.isLandOwned = false
    self.lastColourIndex = 0

    self.lastSaplingTooClose = false
    self.plantingPermitted = false

    self.canPlantTree = false
    self.isPlanting = false
    self.saplingPlanted = false

    self.standardActivate = true

    self.isMultiplayer = g_currentMission.missionDynamicInfo.isMultiplayer

    return self
end

function HandPlantSaplingsShovel:load(xmlFilename, player)
    if not HandPlantSaplingsShovel:superClass().load(self, xmlFilename, player) then
        return false
    end

    local xmlFile = loadXMLFile("TempXML", xmlFilename)

    self.shovelNode = I3DUtil.indexToObject(self.rootNode, getXMLString(xmlFile, "handTool.handPlantSaplings.shovelAnimation#node"))
    if self.shovelNode ~= nil then
        local animKey = "handTool.handPlantSaplings.shovelAnimation"

        self.shovelAnimation = {part = {}}
        self.shovelAnimation.duration = Utils.getNoNil(getXMLFloat(xmlFile, animKey .. "#duration"), 4) * 1000

        if self.shovelAnimation.duration == 0 then
            self.shovelAnimation.duration = 1000
        end

        self.shovelAnimation.time = 0
        self.shovelAnimation.direction = 0

        self.shovelAnimation.part.animCurve = AnimCurve:new(linearInterpolatorN)

        local missingFrames = true

        local j = 0
        while true do
            local frameKey = string.format("%s.keyFrame(%d)", animKey, j)
            if not hasXMLProperty(xmlFile, frameKey) then
                break
            end

            local keyTime = getXMLFloat(xmlFile, frameKey .. "#time")
            local keyframe = {time = keyTime}

            local rx,ry,rz = StringUtil.getVectorFromString(getXMLString(xmlFile, frameKey .. "#rotation"))
            local x,y,z = StringUtil.getVectorFromString(getXMLString(xmlFile, frameKey .. "#translation"))
            local sx,sy,sz = StringUtil.getVectorFromString(getXMLString(xmlFile, frameKey .. "#scale"))

            local dx,dy,dz = getTranslation(self.shovelNode)
            keyframe[1] = Utils.getNoNil(x, dx)
            keyframe[2] = Utils.getNoNil(y, dy)
            keyframe[3] = Utils.getNoNil(z, dz)

            local drx,dry,drz = getRotation(self.shovelNode)
            keyframe[4] = Utils.getNoNilRad(rx, drx)
            keyframe[5] = Utils.getNoNilRad(ry, dry)
            keyframe[6] = Utils.getNoNilRad(rz, drz)

            local dsx,dsy,dsz = getScale(self.shovelNode)
            keyframe[7] = Utils.getNoNil(sx, dsx)
            keyframe[8] = Utils.getNoNil(sy, dsy)
            keyframe[9] = Utils.getNoNil(sz, dsz)

            self.shovelAnimation.part.animCurve:addKeyframe(keyframe)

            missingFrames = false

            j = j + 1
        end

        if missingFrames then
            self.shovelNode = nil
            self.shovelAnimation = nil

            g_logManager:xmlWarning(xmlFilename, "No animation was found, this is a minimum requirement for the shovel.")
            return false
        end
    else
        g_logManager:xmlWarning(xmlFilename, "No animation node was found, this is a minimum requirement for the shovel.")
        return false
    end

    self.diggingSample = g_soundManager:loadSampleFromXML(xmlFile, "handTool.handPlantSaplings.sounds", "digging", self.baseDirectory, self.rootNode, 1, AudioGroup.ENVIRONMENT, nil, nil)

    local filename = getXMLString(xmlFile, "handTool.handPlantSaplings.placeHolder#file")
    if filename ~= nil then
        local i3dNode = g_i3DManager:loadSharedI3DFile(filename, self.baseDirectory, false, false, false)
        if i3dNode ~= 0 then
            self.placeHolderFilename = filename
            self.placeHolder = getChildAt(i3dNode, 0)
            setVisibility(self.placeHolder, false)
            link(g_currentMission.terrainRootNode, self.placeHolder)
            delete(i3dNode)
        end
    end

    delete(xmlFile)

    return true
end

function HandPlantSaplingsShovel:delete()
    if self.placeHolder ~= nil then
        delete(self.placeHolder)
    end

    if self.placeHolderFilename ~= nil then
        g_i3DManager:releaseSharedI3DFile(self.placeHolderFilename, self.baseDirectory, false)
    end

    if self.diggingSample~= nil then
        g_soundManager:deleteSample(self.diggingSample)
        self.diggingSample = nil
    end

    HandPlantSaplingsShovel:superClass().delete(self)
end

function HandPlantSaplingsShovel:update(dt, allowInput)
    HandPlantSaplingsShovel:superClass().update(self, dt)

    if allowInput and self.storageArea ~= nil then
        local backpackTreeType, backpackFillLevel = self.storageArea:getBackpackLevel()
        if backpackFillLevel > 0 then
            local treeTypeName = self.storageArea:getSaplingTitleFromType(backpackTreeType)
            g_currentMission:addExtraPrintText(string.format("%s %d %s ( %s )", self.i18n:getText("hps_backpackText"), backpackFillLevel, self.i18n:getText("unit_pieces"), treeTypeName))
        else
            g_currentMission:addExtraPrintText(self.i18n:getText("hps_emptyBackpack"))
        end

        self.plantingPermitted = false

        if self.inputWaitTimer > 0 then
            self.inputWaitTimer = self.inputWaitTimer - 1
        end

        local saplingPallet, palletFillLevel = self:getSaplingPalletInRange()
        if saplingPallet == nil then
            if self:isPlayerInTrigger() then
                self.extraInputsActive = true

                local storageSpace = self.storageArea:getStorageSpace()
                local canAdd = storageSpace > 0 and backpackFillLevel > 0

                if canAdd then
                    local eventId = self.player.inputInformation.registrationList[InputAction.ACTIVATE_HANDTOOL].eventId

                    local storageAdd = storageSpace
                    if backpackFillLevel < storageSpace then
                        storageAdd = backpackFillLevel
                    end

                    local treeTypeName = self.storageArea:getSaplingTitleFromType(backpackTreeType)
                    g_inputBinding:setActionEventText(self.storageAddEventId, string.format(self.i18n:getText("hps_storageInput"), storageAdd, treeTypeName))
                end

                g_inputBinding:setActionEventTextVisibility(self.storageAddEventId, canAdd)
                g_inputBinding:setActionEventActive(self.storageAddEventId, true)

                local backpackSpace = self.storageArea:getBackpackSpace()
                local storageTreeType, storageFillLevel = self.storageArea:getStorageLevel()
                local canRemove = storageFillLevel > 0 and backpackSpace > 0

                if canRemove then
                    local storageRemove = backpackSpace
                    if storageFillLevel < backpackSpace then
                        storageRemove = storageFillLevel
                    end

                    local treeTypeName = self.storageArea:getSaplingTitleFromType(storageTreeType)
                    g_inputBinding:setActionEventText(self.storageRemoveEventId, string.format(self.i18n:getText("hps_storageOutput"), storageRemove, treeTypeName))
                end

                g_inputBinding:setActionEventTextVisibility(self.storageRemoveEventId, canRemove)
                g_inputBinding:setActionEventActive(self.storageRemoveEventId, true)
            else
                if self.extraInputsActive then
                    self.extraInputsActive = false
                    g_inputBinding:setActionEventTextVisibility(self.storageAddEventId, false)
                    g_inputBinding:setActionEventActive(self.storageAddEventId, false)
                    g_inputBinding:setActionEventTextVisibility(self.storageRemoveEventId, false)
                    g_inputBinding:setActionEventActive(self.storageRemoveEventId, false)
                end
            end

            if not self.standardActivate then
                self.standardActivate = true

                local eventId = self.player.inputInformation.registrationList[InputAction.ACTIVATE_HANDTOOL].eventId
                g_inputBinding:setActionEventText(eventId, self.i18n:getText("input_ACTIVATE_HANDTOOL"))
            end

            local x, y, z = localToWorld(self.player.cameraNode, 0, 0, -2.5)
            local dx, dy, dz = localDirectionToWorld(self.player.cameraNode, 0, -1, 0)
            raycastClosest(x, y, z, dx, dy, dz, "findGroundRaycastCallback", 10, self)
        else
            self.standardActivate = false

            local eventId = self.player.inputInformation.registrationList[InputAction.ACTIVATE_HANDTOOL].eventId
            local spaceText = string.format(self.i18n:getText("hps_takeSaplings"), math.min(self.storageArea:getBackpackSpace(), palletFillLevel))
            g_inputBinding:setActionEventText(eventId, spaceText)

            self:setPlaceHolder(false)
        end

        if self.activatePressed then
            if saplingPallet == nil then
                if self.inputWaitTimer <= 0 then
                    if self.plantingPermitted then
                        if self.isLandOwned then
                            if g_treePlantManager:canPlantTree() then
                                if not self.isPlanting then
                                    self:setShovelState(true, true, self.nodeLocation)
                                end

                                if not self.saplingPlanted then
                                    local currentTime = self:setShovelAnimationTime(self.shovelAnimation.time + dt / self.shovelAnimation.duration)

                                    if currentTime == 1 then
                                        self:setShovelState(false, true)
                                        self.saplingPlanted = true

                                        self:setShovelAnimationTime(0.0)

                                        local x, y, z = self.nodeLocation.x, self.nodeLocation.y, self.nodeLocation.z

                                        if self.isServer then
                                            self:plantSapling(backpackTreeType, x, y, z, backpackFillLevel - 1)
                                        else
                                            g_client:getServerConnection():sendEvent(HPS_PlantSaplingEvent:new(self.player, backpackTreeType, x, y, z, backpackFillLevel - 1))
                                        end

                                        self.storageArea.lastPlanted.x = x
                                        self.storageArea.lastPlanted.y = y
                                        self.storageArea.lastPlanted.z = z
                                        self.storageArea.lastPlanted.valid = true

                                        if not self.isServer then
                                            -- MP 'true' client only. (Only for local client but should not be an issue, just a counter anyway.)
                                            table.insert(self.storageArea.plantedTrees, {x = x, y = y, z = z})
                                        end
                                    end
                                end
                            else
                                g_currentMission:showBlinkingWarning(self.i18n:getText("warning_tooManyTrees"))
                            end
                        else
                            g_currentMission:showBlinkingWarning(self.i18n:getText("warning_youDontHaveAccessToThisLand"))
                        end
                    else
                        if not self.saplingPlanted then
                            if self.player.debugFlightMode then
                                g_currentMission:showBlinkingWarning(self.i18n:getText("hps_flightActive"))
                            elseif self.storageArea.backpack.fillLevel <= 0 then
                                g_currentMission:showBlinkingWarning(self.i18n:getText("hps_emptyBackpackWarning"))
                            else
                                local treeTypeName = self.storageArea:getSaplingTitleFromType(backpackTreeType)
                                g_currentMission:showBlinkingWarning(string.format(self.i18n:getText("hps_plantWarning"), treeTypeName))
                            end
                        end
                    end
                end
            else
                if self.inputWaitTimer <= 0 then
                    self.inputWaitTimer = 100

                    local isValid, typeIndex = self:getSaplingPalletIsValid(saplingPallet)
                    if isValid then
                        if self.storageArea:getBackpackSpace() > 0 then
                            local saplingPalletId = NetworkUtil.getObjectId(saplingPallet)
                            if self.isServer then
                                self:setSaplingPalletLevel(saplingPalletId, typeIndex)
                            else
                                g_client:getServerConnection():sendEvent(HPS_PalletInteractionEvent:new(self.player, saplingPalletId, typeIndex))
                            end
                        else
                            g_currentMission:showBlinkingWarning(self.i18n:getText("hps_backpackFull"))
                        end
                    else
                        local treeTypeName = self.storageArea:getSaplingTitleFromType(backpackTreeType)
                        g_currentMission:showBlinkingWarning(string.format(self.i18n:getText("hps_backpackTypeWarning"), treeTypeName))
                    end
                end

                if self.saplingPlanted or self.isPlanting then
                    self.saplingPlanted = false
                    self:setShovelState(false, false)
                    self:setShovelAnimationTime(0.0)
                end
            end
        else
            if self.saplingPlanted or self.isPlanting then
                self.saplingPlanted = false
                self:setShovelState(false, false)
                self:setShovelAnimationTime(0.0)
            end
        end
    end

    self.activatePressed = false
    self:raiseActive()
end

function HandPlantSaplingsShovel:setShovelAnimationTime(currentTime)
    local currentTime = MathUtil.clamp(currentTime, 0, 1)

    local v = self.shovelAnimation.part.animCurve:get(currentTime)

    setTranslation(self.shovelNode, v[1], v[2], v[3])
    setRotation(self.shovelNode, v[4], v[5], v[6])
    setScale(self.shovelNode, v[7], v[8], v[9])

    self.shovelAnimation.time = currentTime

    self:setDiggingSound(currentTime > 0)

    return currentTime
end

function HandPlantSaplingsShovel:setDiggingSound(active)
    if self.diggingSample ~= nil then
        if self.diggingSampleActive ~= active then
            self.diggingSampleActive = active

            if active then
                g_soundManager:playSample(self.diggingSample)
            else
                g_soundManager:stopSample(self.diggingSample)
            end
        end
    end
end

function HandPlantSaplingsShovel:setShovelState(isActive, playerLocked, translation)
    self.isPlanting = isActive

    self.player:lockInput(playerLocked)
    self.player.walkingIsLocked = playerLocked

    if isActive then
        if translation ~= nil then
            if self.currentPlayerHandNode ~= g_currentMission.terrainRootNode then
                self.oldPlayerHandNode = self.currentPlayerHandNode
                self.currentPlayerHandNode = g_currentMission.terrainRootNode

                link(g_currentMission.terrainRootNode, self.handNode)
                local _, ry, _ = getRotation(self.player.cameraNode)
                setTranslation(self.handNode, translation.x, translation.y, translation.z)
                setRotation(self.handNode, 0.5, ry + 60, 0)
                setScale(self.handNode, 1.5, 1.5, 1.5)
            end
        else
            setTranslation(self.handNode, 0, -100, 0)
        end
    else
        if self.oldPlayerHandNode ~= nil then
            if self.currentPlayerHandNode ~= self.oldPlayerHandNode then
                self.currentPlayerHandNode = self.oldPlayerHandNode
                link(self.oldPlayerHandNode, self.handNode)

                setTranslation(self.handNode, self.handNodePosition[1], self.handNodePosition[2], self.handNodePosition[3])
                setRotation(self.handNode, self.handNodeRotation[1], self.handNodeRotation[2], self.handNodeRotation[3])
                setScale(self.handNode, 1, 1, 1)
            end
        end
    end
end

function HandPlantSaplingsShovel:findGroundRaycastCallback(hitObjectId, x, y, z, distance, dx, dy, dz)
    if self.player.debugFlightMode then
        self:setPlaceHolder(false)
        return true
    end

    self.nodeLocation.x = x
    self.nodeLocation.y = y
    self.nodeLocation.z = z

    if self:getIsDistanceValid(x, y, z) then
        local active = not self:isPlayerInTrigger()

        if HandPlantSaplingsShovel.getIsValidGroundType(hitObjectId, x, y, z) and self.storageArea.backpack.fillLevel > 0 then
            if active then
                self.plantingPermitted = true

                if self.isPlanting then
                    self:setPlaceHolder(true, false, 3, x, y, z)
                else
                    self.isLandOwned = g_currentMission.accessHandler:canFarmAccessLand(self.player.farmId, x, z)

                    if self.isLandOwned then
                        self:setPlaceHolder(true, false, 2, x, y, z)
                    else
                        self:setPlaceHolder(true, false, 1, x, y, z)
                    end
                end
            else
                self:setPlaceHolder(active)
            end
        else
            self:setPlaceHolder(active, false, 1, x, y, z)
        end

        self.lastSaplingTooClose = false
    else
        if not self.saplingPlanted then
            self:setPlaceHolder(true, false, 1, x, y, z)
            self.lastSaplingTooClose = true
        else
            self:setPlaceHolder(false)
        end
    end

    return true
end

function HandPlantSaplingsShovel:setPlaceHolder(active, force, colourIndex, x, y, z)
    if self.placeHolder ~= nil then
        if self.placeHolderActive ~= active or force then
            setVisibility(self.placeHolder, active and self.placeHolderToggle)
            self.placeHolderActive = active
        end

        if active and self.placeHolderToggle and not force then
            local rgba = HandPlantSaplingsShovel.INDEX_TO_COLOUR[colourIndex]
            if rgba == nil then
                rgba = HandPlantSaplingsShovel.INDEX_TO_COLOUR[1]
            end

            if colourIndex ~= self.lastColourIndex then
                setShaderParameter(self.placeHolder, "colorScale", rgba[1], rgba[2], rgba[3], rgba[4], false)
                self.lastColourIndex = colourIndex
            end

            local lastSaplingDistance = 0.000
            if self.storageArea.lastPlanted.valid then
                lastSaplingDistance = MathUtil.vector3Length(x - self.storageArea.lastPlanted.x, y - self.storageArea.lastPlanted.y, z - self.storageArea.lastPlanted.z)
            end

            setTranslation(self.placeHolder, x, y, z)

            if g_gameSettings:getValue("useAcre") then
                Utils.renderTextAtWorldPosition(x, y, z, string.format("( %.2f m)", lastSaplingDistance), getCorrectTextSize(0.025), -0.05, rgba)
            else
                Utils.renderTextAtWorldPosition(x, y, z, string.format("( %.2f ft)", lastSaplingDistance * 3.281), getCorrectTextSize(0.025), -0.05, rgba)
            end
        end
    end
end

function HandPlantSaplingsShovel:plantSapling(treeTypeIndex, x, y, z, backpackLevel)
    local yRot = math.random() * 2 * math.pi
    g_treePlantManager:plantTree(treeTypeIndex, x, y, z, 0, yRot, 0, 0)

    local stats = g_farmManager:getFarmById(self.player.farmId).stats
    stats:updateStats("plantedTreeCount", 1)

    self.storageArea:updateBackpack(backpackLevel, self.storageArea.backpack.treeType or 1)
end

function HandPlantSaplingsShovel:getSaplingPalletInRange()
    for _, vehicle in pairs(g_currentMission.vehicles) do
        if vehicle.typeName == "pallet" and g_currentMission.accessHandler:canFarmAccess(self.player.farmId, vehicle) then
            local distance = calcDistanceFrom(self.player.rootNode, vehicle.rootNode)
            if distance < 3.0 then
                local filltype = vehicle:getFillUnitFillType(1)
                if filltype == FillType.TREESAPLINGS then
                    local fillLevel = vehicle:getFillUnitFillLevel(1)
                    if fillLevel > 0 then
                        return vehicle, fillLevel
                    end
                end
            end
        end
    end

    return
end

function HandPlantSaplingsShovel:setSaplingPalletLevel(palletToSet, treeTypeIndex)
    if self.isServer then
        local saplingPallet = NetworkUtil.getObject(palletToSet)
        if saplingPallet ~= nil then
            if treeTypeIndex == nil then
                treeTypeIndex = 1

                local fillType = saplingPallet:getFillUnitFillType(1)
                if fillType == FillType.TREESAPLINGS then
                    local treeTypeName = getUserAttribute(saplingPallet.rootNode, "treeType")
                    if treeTypeName ~= nil then
                        local desc = g_treePlantManager:getTreeTypeDescFromName(treeTypeName)
                        if desc ~= nil then
                            treeTypeIndex = desc.index
                        end
                    end
                end
            end

            local palletFillLevel = saplingPallet:getFillUnitFillLevel(1)
            local levelToFill = self.storageArea:getBackpackSpace()

            if palletFillLevel < levelToFill then
                levelToFill = palletFillLevel
            end

            saplingPallet:addFillUnitFillLevel(self.player.farmId, 1, -levelToFill, FillType.TREESAPLINGS, ToolType.UNDEFINED)

            local _, backpackLevel = self.storageArea:getBackpackLevel()
            self.storageArea:updateBackpack(backpackLevel + levelToFill, treeTypeIndex)
        end
    end
end

function HandPlantSaplingsShovel:getSaplingPalletIsValid(saplingPallet)
    local treeTypeIndex = 1

    local treeTypeName = getUserAttribute(saplingPallet.rootNode, "treeType")
    if treeTypeName ~= nil then
        local desc = g_treePlantManager:getTreeTypeDescFromName(treeTypeName)
        if desc ~= nil then
            treeTypeIndex = desc.index
        end
    end

    if self.storageArea.backpack.treeType > 0 then
        return treeTypeIndex == self.storageArea.backpack.treeType, treeTypeIndex
    end

    return true, treeTypeIndex
end

function HandPlantSaplingsShovel:getIsDistanceValid(x, y, z)
    local growingTrees = g_treePlantManager.treesData.growingTrees

    if not self.isServer then
        growingTrees = self.storageArea.plantedTrees
    end

    local numGrowingTrees = #growingTrees
    if numGrowingTrees > 0 then
        for i = 1, numGrowingTrees do
            local growingTree = growingTrees[i]
            local plantingDistance = MathUtil.vector3Length(x - growingTree.x, y - growingTree.y, z - growingTree.z)
            if plantingDistance < self.storageArea.plantingDistance then
                return false
            end
        end
    end

    return true
end

function HandPlantSaplingsShovel.getIsValidGroundType(hitObjectId, x, y, z)
    if hitObjectId ~= nil and not PlacementUtil.isInsideRestrictedZone(g_currentMission.restrictedZones, nil, x, y, z) then
        if hitObjectId == g_currentMission.terrainRootNode then
            local _, _, _, depth, materialId = getTerrainAttributesAtWorldPos(g_currentMission.terrainRootNode, x, y, z, true, true, true, true, false)
            return depth > 0.8
        end
    end

    return false
end

function HandPlantSaplingsShovel:onDeactivate()
    self.storageArea = nil

    if self.player == g_currentMission.player then
        self:setShovelState(false, false)
        self:setShovelAnimationTime(0.0)

        self.player:lockInput(playerLocked)
        self.player.walkingIsLocked = playerLocked

        self:setPlaceHolder(false, false)

        local eventId = self.player.inputInformation.registrationList[InputAction.ACTIVATE_HANDTOOL].eventId
        g_inputBinding:setActionEventText(eventId, self.i18n:getText("input_ACTIVATE_HANDTOOL"))
    end

    HandPlantSaplingsShovel:superClass().onDeactivate(self)
end

function HandPlantSaplingsShovel:onActivate()
    self.storageArea = self.player.activeSaplingEquipment

    HandPlantSaplingsShovel:superClass().onActivate(self)
end

function HandPlantSaplingsShovel:getIsActiveForInput()
    if self.player == g_currentMission.player and not g_gui:getIsGuiVisible() then
        return true
    end

    return false
end

function HandPlantSaplingsShovel:isPlayerInTrigger()
    return self.storageArea.playerInRange
end

function HandPlantSaplingsShovel:isBeingUsed()
    return self.isPlanting
end

function HandPlantSaplingsShovel:registerActionEvents()
    HandPlantSaplingsShovel:superClass().registerActionEvents(self)

    g_inputBinding:beginActionEventsModification(Player.INPUT_CONTEXT_NAME)

    local eventId = ""

    _, eventId = g_inputBinding:registerActionEvent(InputAction.SHIFT_GEAR_UP, self, self.onStorageAdd, false, true, false, false)
    self.storageAddEventId = eventId

    _, eventId = g_inputBinding:registerActionEvent(InputAction.SHIFT_GEAR_DOWN, self, self.onStorageRemove, false, true, false, false)
    self.storageRemoveEventId = eventId

    g_inputBinding:endActionEventsModification()
end

function HandPlantSaplingsShovel:onStorageAdd(_, inputValue)
    local backpackTreeType, backpackFillLevel = self.storageArea:getBackpackLevel()
    local storageTreeType, storageFillLevel = self.storageArea:getStorageLevel()

    if backpackFillLevel > 0 then
        local storageSpace = self.storageArea:getStorageSpace()
        if storageSpace > 0 then

            local storageAdd = storageSpace
            if backpackFillLevel < storageSpace then
                storageAdd = backpackFillLevel
            end

            if storageTreeType == 0 or storageTreeType == backpackTreeType then
                self.storageArea:updateBackpack(backpackFillLevel - storageAdd, backpackTreeType)
                self.storageArea:updateStorage(storageFillLevel + storageAdd, backpackTreeType)
            else
                local treeTypeName = self.storageArea:getSaplingTitleFromType(backpackTreeType)
                g_currentMission:showBlinkingWarning(string.format(self.i18n:getText("hps_storageTypeWarning"), treeTypeName))
            end
        else
            g_currentMission:showBlinkingWarning(self.i18n:getText("hps_fullStorageWarning"))
        end
    else
        g_currentMission:showBlinkingWarning(self.i18n:getText("hps_backpackEmpty"))
    end
end

function HandPlantSaplingsShovel:onStorageRemove(_, inputValue)
    local backpackTreeType, backpackFillLevel = self.storageArea:getBackpackLevel()
    local storageTreeType, storageFillLevel = self.storageArea:getStorageLevel()
    local backpackSpace = self.storageArea:getBackpackSpace()

    if backpackSpace > 0 then
        if storageFillLevel > 0 then

            local storageRemove = backpackSpace
            if storageFillLevel < backpackSpace then
                storageRemove = storageFillLevel
            end

            if backpackTreeType == 0 or backpackTreeType == storageTreeType then
                self.storageArea:updateBackpack(backpackFillLevel + storageRemove, storageTreeType)
                self.storageArea:updateStorage(storageFillLevel - storageRemove, storageTreeType)
            else
                local treeTypeName = self.storageArea:getSaplingTitleFromType(storageTreeType)
                g_currentMission:showBlinkingWarning(string.format(self.i18n:getText("hps_backpackTypeWarning"), treeTypeName))
            end
        else
            g_currentMission:showBlinkingWarning(self.i18n:getText("hps_emptyStorageWarning"))
        end
    else
        g_currentMission:showBlinkingWarning(self.i18n:getText("hps_backpackFull"))
    end
end

registerHandTool("handPlantSaplingsShovel", HandPlantSaplingsShovel)
