-- variableBaleCapacity specialization for FS19
--
-- Enables a variable bale chamber to create bales with a larger capacity.
-- The actual bale size does not get changed, so all the standard bale collectors will still work
-- Custom bale capacities can be set in modsSettings/VariableBaleCapacity/savegameXX/BaleCapacities.xml
--
-- Author: sperrgebiet
-- Credits: Idea and previous LS versions: Ziuta, 50keda, UnknownModder

VarBaleCap = {}
VarBaleCap.eventName = {}
VarBaleCap.ModName = g_currentModName
VarBaleCap.ModDirectory = g_currentModDirectory
VarBaleCap.Version = "1.0.0.0"

VarBaleCap.debug = fileExists(VarBaleCap.ModDirectory ..'debug')

print(string.format('VarBaleCap v%s - DebugMode %s)', VarBaleCap.Version, tostring(VarBaleCap.debug)))

--Set some defaults; 20k might seem odd, but it's required for cotton modules
VarBaleCap.Capacities = {1500,2000,4000,6000,8000,10000, 20000}

addModEventListener(VarBaleCap)

function VarBaleCap:dp(val, fun, msg) -- debug mode, write to log
	if not VarBaleCap.debug then
		return;
	end

	if msg == nil then
		msg = ' ';
	else
		msg = string.format(' msg = [%s] ', tostring(msg));
	end

	local pre = 'VarBaleCap DEBUG:';

	if type(val) == 'table' then
		--if #val > 0 then
			print(string.format('%s BEGIN Printing table data: (%s)%s(function = [%s()])', pre, tostring(val), msg, tostring(fun)));
			DebugUtil.printTableRecursively(val, '.', 0, 3);
			print(string.format('%s END Printing table data: (%s)%s(function = [%s()])', pre, tostring(val), msg, tostring(fun)));
		--else
		--	print(string.format('%s Table is empty: (%s)%s(function = [%s()])', pre, tostring(val), msg, tostring(fun)));
		--end
	else
		print(string.format('%s [%s]%s(function = [%s()])', pre, tostring(val), msg, tostring(fun)));
	end
end

function VarBaleCap.prerequisitesPresent(specializations)
    return SpecializationUtil.hasSpecialization(Baler, specializations)
end

function VarBaleCap:loadMap(name)
	print("--- loading VarBaleCap V".. VarBaleCap.Version .. " | ModName " .. VarBaleCap.ModName .. " ---")

	-- Just load the config on the server
	if g_server ~= nil then
		VarBaleCap.userPath = getUserProfileAppPath();
		VarBaleCap.saveBasePath = VarBaleCap.userPath .. 'modsSettings/VariableBaleCapacity/';
		VarBaleCap.savePath = VarBaleCap.saveBasePath .. 'savegame' .. g_careerScreen.selectedIndex .. '/';
		createFolder(VarBaleCap.userPath .. 'modsSettings/');
		createFolder(VarBaleCap.saveBasePath);
		createFolder(VarBaleCap.savePath);
		VarBaleCap.xmlFilename = VarBaleCap.savePath .. 'BaleCapacities.xml';	
		
		VarBaleCap:loadConfig()
	end
end

function VarBaleCap.registerEventListeners(vehicleType)
	local functionNames = {	"onRegisterActionEvents", "onDraw", "onLoad", "saveToXMLFile", "onReadStream", "onWriteStream" }
	
	for _, functionName in ipairs(functionNames) do
		SpecializationUtil.registerEventListener(vehicleType, functionName, VarBaleCap)
	end
end

function VarBaleCap:onRegisterActionEvents(isSelected, isOnActiveVehicle)
	
	local spec = self.spec_varBaleCap	
	spec.actionEvents = {}
	if isSelected then
		local _, actionEventId = self:addActionEvent(spec.actionEvents, 'varBaleCap_Change', self, VarBaleCap.action_varBaleCap_Change, false, true, false, true, nil)
		g_inputBinding:setActionEventTextPriority(actionEventId, GS_PRIO_HIGH)
	end		
end

function VarBaleCap:action_varBaleCap_Change(actionName, keyStatus, arg3, arg4, arg5)
	if table.getn(self.spec_baler.bales) > 0 then
		g_currentMission:showBlinkingWarning(g_i18n.modEnvironments[VarBaleCap.ModName].texts.WarningBaleFinished, 5000)
	else
		VarBaleCap:updateCapacity(self, false)
	end
end

function VarBaleCap:onDraw(isActiveForInput, isActiveForInputIgnoreSelection, isSelected)
	g_currentMission:addExtraPrintText(g_i18n.modEnvironments[VarBaleCap.ModName].texts.VarBaleCapCurrent .. string.format(" : %d",  self:getFillUnitCapacity(self.spec_varBaleCap.fillUnitIndex)));
end

function VarBaleCap:onLoad(savegame)
	VarBaleCap:getLargestFillUnit(self)
	self.spec_varBaleCap.capIndex = Utils.getNoNil(VarBaleCap:getIndexByValue(VarBaleCap.Capacities, self:getFillUnitCapacity(self.spec_varBaleCap.fillUnitIndex)), 1)
	
	if savegame ~= nil then
		local xmlFile = savegame.xmlFile
		local key = savegame.key..".varBaleCap#baleCapacity"
		
		--It's possible that a baler uses an unknown capacity or that we dropped one from our desired capacities. I that case we want to default to some values
		local varBaleCap = Utils.getNoNil(getXMLInt(xmlFile, key), 4000)
		local capIndex = Utils.getNoNil(VarBaleCap:getIndexByValue(VarBaleCap.Capacities, varBaleCap), 1)
		self.spec_varBaleCap.capIndex = capIndex
		VarBaleCap:setBaleCapacity(self, capIndex, true)
	end
end

function VarBaleCap:saveToXMLFile(xmlFile, key)
	if self.spec_varBaleCap ~= nil then
		if self.spec_varBaleCap.fillUnitIndex ~= nil then
			setXMLInt(xmlFile, key.."#baleCapacity", self:getFillUnitCapacity(self.spec_varBaleCap.fillUnitIndex))
		end
	end
end

function VarBaleCap:updateCapacity(obj, noEventSend)
	local nextCapIndex = 1

	--Get current capacityIndex. When the current baler capacity is not in our defined list we'll default to the first capacity in the list
	local currCapIndex = self:getIndexByValue(self.Capacities, obj:getFillUnitCapacity(obj.spec_varBaleCap.fillUnitIndex))
	if currCapIndex == nil then
		nextCapIndex = 1
	elseif (currCapIndex + 1) <= #self.Capacities then
		nextCapIndex = currCapIndex + 1
	end
	
	--In case the current fillLevel is larger than the desired nextCapIndex we'll find the next suitable one
	local currentFillLevel = self:getFillLevel(obj)	

	if self.Capacities[nextCapIndex] < currentFillLevel then
		for k, v in ipairs(self.Capacities) do
			if currentFillLevel < v then
				nextCapIndex = k
				break
			end
		end
	end
	
	-- If we can't find a suitable new capacity we've to finish the bale first
	if self.Capacities[nextCapIndex] == obj:getFillUnitCapacity(obj.spec_varBaleCap.fillUnitIndex) then
		g_currentMission:showBlinkingWarning(g_i18n.modEnvironments[VarBaleCap.ModName].texts.WarningFillLevel, 5000)
	else
		self:setBaleCapacity(obj, nextCapIndex, noEventSend)
	end

end

function VarBaleCap:getLargestFillUnit(obj)
	-- For balers with pre-chambers we've multiple fillUnits. So we'll always use the largest fillUnit for our capacity.
	local fillUnits = obj:getFillUnits();
	if #fillUnits > 1 then
		local largestCap = 0
		for _, v in pairs(fillUnits) do
			local index = v.fillUnitIndex
            local capacity = obj:getFillUnitCapacity(index)
			if capacity >= largestCap then
				largestCap = capacity
				obj.spec_varBaleCap.fillUnitIndex = index
			end
		end
	end
	
	--Set a default
	if obj.spec_varBaleCap.fillUnitIndex == nil then
		obj.spec_varBaleCap.fillUnitIndex = 1
	end
end

function VarBaleCap:getIndexByValue(tbl, val)
	for k, v in ipairs(tbl) do
		if v == val then
			return k
		end
	end
end

function VarBaleCap:getFillLevel(obj)
	return obj:getFillUnitFillLevel(obj.spec_varBaleCap.fillUnitIndex)
end

function VarBaleCap:setBaleCapacity(obj, capIndex, noEventSend)
	obj:setFillUnitCapacity(obj.spec_varBaleCap.fillUnitIndex, self.Capacities[capIndex])
	obj.spec_varBaleCap.capIndex = capIndex
	VarBaleCapEvent.sendEvent(obj, capIndex, noEventSend)
end

function VarBaleCap:loadConfig()
	local capacities = {}
	if fileExists(self.xmlFilename) then
		self.saveFile = loadXMLFile('VarBaleCap.loadFile', self.xmlFilename)
		
		if hasXMLProperty(self.saveFile, 'BaleCapacities') then
			local savedCaps = getXMLString(self.saveFile, 'BaleCapacities')
			
			for str in string.gmatch(savedCaps, '([^,]+)') do
			   table.insert(capacities, tonumber(str))
			end
			
			if #capacities > 0 then 
				table.sort(capacities)
				self.Capacities = capacities
				print("VarBaleCap: Bale capacities loaded from XML")
			end
		end
	else
		-- If no config exists yet, we'll just save the default one
		self:saveConfig()
	end
end

function VarBaleCap:saveConfig()
	self.saveFile = createXMLFile('VarBaleCap.saveFile', self.xmlFilename, 'BaleCapacities')

	setXMLString(self.saveFile, 'BaleCapacities', table.concat(self.Capacities, ","))

	saveXMLFile(self.saveFile);

	print("VarBaleCap: Bale capacities saved");
end

function VarBaleCap:onReadStream(streamId, connection)
	local capIndex = streamReadInt8(streamId)
	VarBaleCap:setBaleCapacity(self, capIndex, true)
	
	local capacities = {}
	local capacitiesString = streamReadString(streamId)
	for str in string.gmatch(capacitiesString, '([^,]+)') do
	   table.insert(capacities, tonumber(str))
	end

	VarBaleCap.Capacities = capacities
end

function VarBaleCap:onWriteStream(streamId, connection)
	streamWriteInt8(streamId, self.spec_varBaleCap.capIndex)
	streamWriteString(streamId, table.concat(VarBaleCap.Capacities, ","))
end


-- MP Events

VarBaleCapEvent = {}
VarBaleCapEvent_mt = Class(VarBaleCapEvent, Event)

InitEventClass(VarBaleCapEvent, "VarBaleCapEvent")

function VarBaleCapEvent:emptyNew()
    local self = Event:new(VarBaleCapEvent_mt)
	self.className = "VarBaleCapEvent"
    return self
end

function VarBaleCapEvent:new(object, capIndex)
    local self = VarBaleCapEvent:emptyNew()
    self.object = object
	self.capIndex = capIndex
    return self
end

function VarBaleCapEvent:readStream(streamId, connection)
	self.object = NetworkUtil.readNodeObject(streamId)
	self.capIndex  = streamReadInt8(streamId)
	self:run(connection)
end

function VarBaleCapEvent:writeStream(streamId, connection)
	NetworkUtil.writeNodeObject(streamId, self.object)
	streamWriteInt8(streamId, self.capIndex)
end

function VarBaleCapEvent:run(connection)
	VarBaleCap:setBaleCapacity(self.object, self.capIndex, true)
    if not connection:getIsServer() then
        g_server:broadcastEvent(VarBaleCapEvent:new(self.object, self.capIndex), nil, connection, self.object)
    end
end

function VarBaleCapEvent.sendEvent(vehicle, capIndex, noEventSend)
	if noEventSend == nil or noEventSend == false then
		if g_server ~= nil then
			g_server:broadcastEvent(VarBaleCapEvent:new(vehicle, capIndex), nil, nil, vehicle)
		else
			g_client:getServerConnection():sendEvent(VarBaleCapEvent:new(vehicle, capIndex))
		end
	end
end