-- 
-- Workshop Tabber
-- 
-- @Interface: 1.1.3.0
-- @Author: LordBanana
-- @Date: 23.05.2019
-- @Version: 1.1.0.0
-- 
-- @Support: LordBanana
-- 
local version = "1.1.0.0 (23.05.2019)";

WorkshopTabber = {};
WorkshopTabber.dir = g_currentModDirectory;
addModEventListener(WorkshopTabber);

-- Temporary variable to store the owner for the DirectSellDialog, since the owner of
-- DirectSellDialog seems to be reset after repairing a vehicle.
WorkshopTabber.tempWorkshop = nil;	

------------------------------------------------
-- @Function: WorkshopTabber:wtOnActivateObject(superFunc)
--
-- @Description: This function will overwrite the default
--				 onActivateObject function in VehicleSellingPoint class.
--			     This function will try to find the first drivable vehicle
--               in the workshop and then calls showNextVehicle to show it.
------------------------------------------------
function WorkshopTabber:wtOnActivateObject(superFunc)
	local found = false;
	local vehicleId = nil;
	
	-- On activation, find first drivable equipment
	for v_Id, inRange in pairs(self.vehicleInRange) do
		local vehicle = g_currentMission.nodeToObject[v_Id];
		
		if vehicle ~= nil then
			for _, specialization in pairs(vehicle.specializationNames) do
				if specialization == "drivable" then
					found = true;
					vehicleId = v_Id;
					break;
				end
			end
		end
		
		if found == true then
			break;
		end
	end
	
	if found == false then
		-- Found no drivable. Start list from zero
		self.currentVehicleId = 0;
		self:showNextVehicle();
	else
		-- Found a drivable! Start list from this one.
		self:showNextVehicle(vehicleId);
	end
end
VehicleSellingPoint.onActivateObject = Utils.overwrittenFunction(VehicleSellingPoint.onActivateObject, WorkshopTabber.wtOnActivateObject);

------------------------------------------------
-- @Function: VehicleSellingPoint:showNextVehicle()
--
-- @Description: This function is added to VehicleSellingPoint class
-- 				 to determine and show the next vehicle that is 
--				 within range of the workshop.
--          	 In case no valid vehicle is found, nil is sent to the
--				 showDialog function which will show the default empty dialog.
--
-- @Arguments:
--    vehicleID  If nil, then this function will select and display next item in VehicleInRange list.
--               If a valid vehicleID, then that vehicle will be displayed.
------------------------------------------------
function VehicleSellingPoint:showNextVehicle(vehicleId)

	-- Depending on if vehicle is nil, do separate things
	if vehicleId ~= nil then
		self.currentVehicle = g_currentMission.nodeToObject[vehicleId];
		if self.currentVehicle ~= nil then
			-- Just show the vehicle received as argument.
			self.currentVehicleId = vehicleId;
			g_gui:showDirectSellDialog({vehicle=self.currentVehicle, owner=self, ownWorkshop=self.ownWorkshop})
			return;
		end
	end
	
	-- Ok, we need to find next vehicle in list.
	
	-- For some reasons, there might be duplicates of the vehicles in the Workshop.vehicleInRange table.
	-- Remove those first, if any.
	for vehicleId, inRange in pairs(self.vehicleInRange) do
		local curr_veh = g_currentMission.nodeToObject[vehicleId]
		local appearances = 0;
				
		for vehicleId2, inRange2 in pairs(self.vehicleInRange) do
			if curr_veh == g_currentMission.nodeToObject[vehicleId2] then
				appearances = appearances + 1;
				
				if appearances > 1 then
					self.vehicleInRange[vehicleId2] = nil;
				end
			end
		end
	end
	
	local newVehicleId, inRange;

	-- Now find next vehicle in list
	if self.currentVehicleId == nil or self.vehicleInRange[self.currentVehicleId] == nil then
		-- No previous vehicle, start list from beginning
		newVehicleId, inRange = next(self.vehicleInRange);
	else
		newVehicleId, inRange = next(self.vehicleInRange, self.currentVehicleId);
	end
	
	-- Just a loop counter to keep track so we don't get stuck in an endless loop.
	local bail_out = 0;
	
	-- Loop to find a valid vehicle in range of the workshop.
	while true do	
		if newVehicleId ~= nil and inRange ~= nil then
			-- newVehicleId found and inRange is valid
			self.currentVehicle = g_currentMission.nodeToObject[newVehicleId];
			if self.currentVehicle ~= nil then
				self.currentVehicleId = newVehicleId;
				break;
			end
		elseif newVehicleId ~= nil and inRange == nil then
			-- Entry found in table, but inRange is nil, remove this
			-- (this sanity check is performed by default
			-- VehicleSellingPoint:determineCurrentVehicle function)
			self.vehicleInRange[newVehicleId] = nil;
		elseif WorkshopTabber:tablelength(self.vehicleInRange) <= 0 then
			-- Empty table, set currentVehicle to nil
			self.currentVehicle = nil;
			self.currentVehicleId = 0;
			break;
		end
		
		-- Bail out is just a last "watchdog" so that we don't get stuck
		-- in an endless loop. 50 turns seems to be a good limit.
		bail_out = bail_out + 1;
		if bail_out >= 50 then
			print("VehicleSellingPoint:showNextVehicle(): Too many loops! Bail out!");
			break;
		end
		
		-- Now find next item in list since the last one was nil.
		if newVehicleId == nil or self.vehicleInRange[newVehicleId] == nil then
			newVehicleId, inRange = next(self.vehicleInRange);
		else
			newVehicleId, inRange = next(self.vehicleInRange, newVehicleId);
		end

	end -- while true do
		
	-- Show the dialog
	g_gui:showDirectSellDialog({vehicle=self.currentVehicle, owner=self, ownWorkshop=self.ownWorkshop})
end;

------------------------------------------------
-- @Function: WorkshopTabber:wtkeyEvent(unicode, sym, modifier, isDown)
--
-- @Description: This function is appended to DirectSellDialog.keyEvent function
-- 				 to catch the 'tab' key.
------------------------------------------------
function WorkshopTabber:wtkeyEvent(unicode, sym, modifier, isDown)

	if sym == 9 and isDown == true then	-- 9 == TAB
	
		--First find the workshop used
		local workshop = self.owner;
	
		if workshop ~= nil then
			workshop:showNextVehicle();
		end
	end
	
end
DirectSellDialog.keyEvent = Utils.appendedFunction(DirectSellDialog.keyEvent, WorkshopTabber.wtkeyEvent);

------------------------------------------------
-- @Function: WorkshopTabber:wtOnVehicleRepairEvent1()
--
-- @Description: This function is prepended to DirectSellDialog.onVehicleRepairEvent function
-- 				 to store the owner of the DirectSellDialog prior to repairing a vehicle.
------------------------------------------------
function WorkshopTabber:wtOnVehicleRepairEvent1()
	-- Store the owner of the DirectSellDialog.
	WorkshopTabber.tempWorkshop = self.owner;
end
DirectSellDialog.onVehicleRepairEvent = Utils.prependedFunction(DirectSellDialog.onVehicleRepairEvent, WorkshopTabber.wtOnVehicleRepairEvent1);

------------------------------------------------
-- @Function: WorkshopTabber:wtOnVehicleRepairEvent2()
--
-- @Description: This function is appended to DirectSellDialog.onVehicleRepairEvent function
-- 				 to restore the owner of the DirectSellDialog after repairing a vehicle.
------------------------------------------------
function WorkshopTabber:wtOnVehicleRepairEvent2()
	-- Restore the owner of the DirectSellDialog.
	self.owner = WorkshopTabber.tempWorkshop;
end
DirectSellDialog.onVehicleRepairEvent = Utils.appendedFunction(DirectSellDialog.onVehicleRepairEvent, WorkshopTabber.wtOnVehicleRepairEvent2);

------------------------------------------------
-- @Function: WorkshopTabber:tablelength(T)
--
-- @Description: Function for getting the length of a table.
------------------------------------------------
function WorkshopTabber:tablelength(T)
  local count = 0
  for _ in pairs(T) do count = count + 1 end
  return count
end

-- Just empty function definitions --
function WorkshopTabber:loadMap() end;
function WorkshopTabber:loadSavegame() end;
function WorkshopTabber:saveSavegame() end;

function WorkshopTabber:update(dt) end;
function WorkshopTabber:keyEvent(unicode, sym, modifier, isDown) end;
function WorkshopTabber:mouseEvent(posX, posY, isDown, isUp, button) end;
function WorkshopTabber:draw() end;
function WorkshopTabber:delete()end;
function WorkshopTabber:deleteMap() end;