--[[
Interface: 1.4.1.0 b5334

Copyright (C) GtX (Andy), 2018

Author: GtX | Andy
Date: 15.03.2018
Version: 1.0.0.0

Contact:
https://forum.giants-software.com
https://github.com/GtX-Andy

History:
V 1.0.0.0 @ 15.03.2018 - FS17 Release
V 1.1.0.0 @ 28.08.2019 - FS19 Release

Important:
No changes are to be made to this script without permission from GtX | Andy
An diesem Skript dürfen ohne Genehmigung von GtX | Andy keine Änderungen vorgenommen werden
]]


HandPlantSaplings = {}

local HandPlantSaplings_mt = Class(HandPlantSaplings, Placeable)
InitObjectClass(HandPlantSaplings, "HandPlantSaplings")

HandPlantSaplings.WARNING_TIME = 4000

function HandPlantSaplings:new(isServer, isClient, customMt)
    local self = Placeable:new(isServer, isClient, customMt or HandPlantSaplings_mt)

    registerObjectClassName(self, "HandPlantSaplings")

    self.i18n = g_i18n
    self.totalFillTypeSellPrice = 0

    self.activateText = self.i18n:getText("ui_details")
    self.playerInRange = false
    self.interactionTrigger = nil
    self.actionEventActive = false

    self.equipmentInUse = false
    self.currentUser = nil
    self.currentUserName = "N/A"
    self.adminUsersOnly = false

    self.storage = {}
    self.storage.treeType = 0
    self.storage.fillLevel = 0
    self.storage.capacity = 0

    self.backpack = {}
    self.backpack.treeType = 0
    self.backpack.fillLevel = 0
    self.backpack.capacity = 0

    self.visibilityNodes = {}

    self.plantedTrees = {} -- MP Only
    self.plantingDistance = 2
    self.lastPlanted = {
        x = 0,
        y = 0,
        z = 0,
        valid = false
    }

    return self
end

function HandPlantSaplings:load(xmlFilename, x,y,z, rx,ry,rz, initRandom)
    if not HandPlantSaplings:superClass().load(self, xmlFilename, x,y,z, rx,ry,rz, initRandom) then
        return false
    end

    local key = "placeable.handPlantSaplings"
    local xmlFile = loadXMLFile("TempXML", xmlFilename)

    self.interactionTrigger = I3DUtil.indexToObject(self.nodeId, getXMLString(xmlFile, key .. ".interactionTrigger#node"))
    if self.interactionTrigger ~= nil then
        local handtoolFileName = getXMLString(xmlFile, key .. ".handtool#filename")
        self.handtoolFileName = Utils.getFilename(handtoolFileName, self.baseDirectory)
        if self.handtoolFileName ~= nil and fileExists(self.handtoolFileName) then
            self.storage.capacity = Utils.getNoNil(getXMLInt(xmlFile, key .. "#storageCapacity"), 10)
            self.backpack.capacity = Utils.getNoNil(getXMLInt(xmlFile, key .. "#backpackCapacity"), 10)

            if hasXMLProperty(xmlFile, key .. ".visibilityNodes") then
                local equipmentNode = I3DUtil.indexToObject(self.nodeId, getXMLString(xmlFile, key .. ".visibilityNodes#equipment"))
                if equipmentNode ~= nil then
                    setVisibility(equipmentNode, true)
                    self.visibilityNodes.equipment = equipmentNode
                end

                local saplingsBackpack = self:loadVisibilityNodes(xmlFile, key .. ".visibilityNodes#saplingsBackpack", xmlFilename)
                if saplingsBackpack ~= nil then
                    self.visibilityNodes.saplingsBackpack = saplingsBackpack
                end

                local saplingsStorage = self:loadVisibilityNodes(xmlFile, key .. ".visibilityNodes#saplingsStorage", xmlFilename)
                if saplingsStorage ~= nil then
                    self.visibilityNodes.saplingsStorage = saplingsStorage
                end
            end
        else
            if self.handtoolFileName == nil then
                g_logManager:xmlWarning(xmlFilename, "No 'handTool' filename given!")
            else
                g_logManager:xmlWarning(xmlFilename, "'handTool' %s could not be found!", handtoolFileName)
            end
        end
    else
        g_logManager:xmlWarning(xmlFilename, "No 'interactionTrigger' was found!")
        return false
    end

    delete(xmlFile)
    xmlFile = nil

    return true
end

function HandPlantSaplings:loadVisibilityNodes(xmlFile, key, xmlFilename)
    local group
    local groupNode = I3DUtil.indexToObject(self.nodeId, getXMLString(xmlFile, key))

    if groupNode ~= nil then
        local numberNodes = getNumOfChildren(groupNode)
        if numberNodes > 0 then
            group = {groupNode = groupNode, nodes = {}}

            for i = 0, numberNodes - 1 do
                local node = getChildAt(groupNode, i)
                setVisibility(node, false)
                table.insert(group.nodes, node)
            end
        else
            g_logManager:xmlWarning(xmlFilename, "No child nodes found in transformGroup at %s.", key)
        end
    end

    return group
end

function HandPlantSaplings:finalizePlacement()
    HandPlantSaplings:superClass().finalizePlacement(self)

    if self.interactionTrigger ~= nil then
        addTrigger(self.interactionTrigger, "interactionTriggerCallback", self)
    end

    g_messageCenter:subscribe(MessageType.PLAYER_FARM_CHANGED, self.onPlayerFarmChanged, self)
end

function HandPlantSaplings:delete()
    if self.interactionTrigger ~= nil then
        removeTrigger(self.interactionTrigger)
        self.interactionTrigger = nil
    end

    if self.playerInRange then
        self.playerInRange = false
        g_currentMission:removeActivatableObject(self)
    end

    self:updateActionEvents(false, true)

    g_messageCenter:unsubscribeAll(self)

    unregisterObjectClassName(self)
    HandPlantSaplings:superClass().delete(self)
end

function HandPlantSaplings:readStream(streamId, connection)
    HandPlantSaplings:superClass().readStream(self, streamId, connection)

    if connection:getIsServer() then
        local backLevel = streamReadInt8(streamId)
        local backTreeType = streamReadInt8(streamId)
        self:updateBackpack(backLevel, backTreeType, true)

        local storeLevel = streamReadInt8(streamId)
        local storeTreeType = streamReadInt8(streamId)
        self:updateStorage(storeLevel, storeTreeType, true)

        local equipmentInUse = streamReadBool(streamId)
        if equipmentInUse then
            local currentUser = NetworkUtil.readNodeObject(streamId)
            self:setEquipmentUsed(equipmentInUse, currentUser, true)
        end

        self.adminUsersOnly = streamReadBool(streamId)
    end
end

function HandPlantSaplings:writeStream(streamId, connection)
    HandPlantSaplings:superClass().writeStream(self, streamId, connection)

    if not connection:getIsServer() then
        streamWriteInt8(streamId, self.backpack.fillLevel)
        streamWriteInt8(streamId, self.backpack.treeType)

        streamWriteInt8(streamId, self.storage.fillLevel)
        streamWriteInt8(streamId, self.storage.treeType)

        local equipmentInUse = self.equipmentInUse and self.currentUser ~= nil
        if streamWriteBool(streamId, equipmentInUse) then
            NetworkUtil.writeNodeObject(streamId, self.currentUser)
        end

        streamWriteBool(streamId, self.adminUsersOnly)
    end
end

function HandPlantSaplings:loadFromXMLFile(xmlFile, key, resetVehicles)
    if not HandPlantSaplings:superClass().loadFromXMLFile(self, xmlFile, key, resetVehicles) then
        return false
    end

    local userSetting = getXMLBool(xmlFile, key .. ".handPlantSaplings.userSetting#adminUsersOnly")
    if userSetting ~= nil then
        self.adminUsersOnly = userSetting
    end

    local backpackLevel = getXMLInt(xmlFile, key .. ".handPlantSaplings.backpack#fillLevel")
    local backpackTreeType = getXMLInt(xmlFile, key .. ".handPlantSaplings.backpack#treeType")
    if backpackLevel ~= nil and backpackTreeType ~= nil then
        local treeTypeIndex = self:getValidTreeTypeIndex(backpackTreeType, key .. ".handPlantSaplings.backpack#treeType")
        self:updateBackpack(math.min(backpackLevel, self.backpack.capacity), treeTypeIndex)
    end

    local storageLevel = getXMLInt(xmlFile, key .. ".handPlantSaplings.storage#fillLevel")
    local storageTreeType = getXMLInt(xmlFile, key .. ".handPlantSaplings.storage#treeType")
    if storageLevel ~= nil and storageTreeType ~= nil then
        local treeTypeIndex = self:getValidTreeTypeIndex(storageTreeType, key .. ".handPlantSaplings.storage#treeType")
        self:updateStorage(math.min(storageLevel, self.storage.capacity), treeTypeIndex)
    end

    return true
end

function HandPlantSaplings:saveToXMLFile(xmlFile, key, usedModNames)
    HandPlantSaplings:superClass().saveToXMLFile(self, xmlFile, key, usedModNames)

    setXMLBool(xmlFile, key .. ".handPlantSaplings.userSetting#modName", self.adminUsersOnly)

    setXMLInt(xmlFile, key .. ".handPlantSaplings.backpack#fillLevel", self.backpack.fillLevel)
    setXMLInt(xmlFile, key .. ".handPlantSaplings.backpack#treeType", self.backpack.treeType)

    setXMLInt(xmlFile, key .. ".handPlantSaplings.storage#fillLevel", self.storage.fillLevel)
    setXMLInt(xmlFile, key .. ".handPlantSaplings.storage#treeType", self.storage.treeType)
end

function HandPlantSaplings:update(dt)
    local player = g_currentMission.player

    if self:getCanInteract(player) then
        if self.storage.fillLevel > 0 then
            local treeTypeName = self:getSaplingTitleFromType(self.storage.treeType)
            g_currentMission:addExtraPrintText(string.format("%s %d %s ( %s )", self.i18n:getText("hps_storageStatus"), self.storage.fillLevel, self.i18n:getText("unit_pieces"), treeTypeName))
        else
            g_currentMission:addExtraPrintText(string.format("%s %d %s", self.i18n:getText("hps_storageStatus"), self.storage.fillLevel, self.i18n:getText("unit_pieces")))
        end

        if self.equipmentInUse then
            if self.currentUser == player then
                self.activateText = self.i18n:getText("hps_return")
            else
                self.activateText = self.i18n:getText("ui_details")
                g_currentMission:addExtraPrintText(string.format(self.i18n:getText("hps_inUseByOther"), self.currentUserName))
            end

        else
            self.activateText = self.i18n:getText("hps_take")

            if not player:hasHandtoolEquipped() then
                if self.backpack.fillLevel > 0 then
                    local treeTypeName = self:getSaplingTitleFromType(self.backpack.treeType)
                    g_currentMission:addExtraPrintText(string.format("%s %d %s ( %s )", self.i18n:getText("hps_equipmentStatus"), self.backpack.fillLevel, self.i18n:getText("unit_pieces"), treeTypeName))
                else
                    g_currentMission:addExtraPrintText(string.format("%s %d %s", self.i18n:getText("hps_equipmentStatus"), self.backpack.fillLevel, self.i18n:getText("unit_pieces")))
                end
            end
        end

        self:raiseActive()
    end
end

function HandPlantSaplings:setEquipmentUsed(equipmentInUse, currentUser, noEventSend)
    HPS_SetEquipmentEvent.sendEvent(self, equipmentInUse, currentUser, noEventSend)

    if equipmentInUse then
        self.equipmentInUse = true

        self.currentUser = currentUser
        self.currentUserName = self:getPlayerName(currentUser)

        if currentUser ~= nil then
            currentUser.activeSaplingEquipment = self

            if currentUser == g_currentMission.player then
                -- Farm:addHandTool(xmlFilename)
                -- No information on what this function does. Are they synced?
                local farm = g_farmManager:getFarmById(currentUser.farmId)
                if farm ~= nil then
                    table.insert(farm.handTools, self.handtoolFileName)
                end

                currentUser:equipHandtool(self.handtoolFileName, false, false)
            end

            if self.isServer then
                currentUser:addDeleteListener(self, "onPlayerDelete")
            end
        end
    else
        self.equipmentInUse = false

        self.currentUser = nil
        self.currentUserName = "N/A"

        if currentUser ~= nil then
            currentUser.activeSaplingEquipment = nil

            if currentUser == g_currentMission.player then
                if self:getEquipmentIsActive(currentUser) then
                    currentUser:equipHandtool("", true)
                end

                -- Farm:removeHandTool(xmlFilename)
                -- No information on what this function does. Are they synced?
                local farm = g_farmManager:getFarmById(currentUser.farmId)
                if farm ~= nil then
                    for key, filename in pairs (farm.handTools) do
                        if filename == self.handtoolFileName then
                            table.remove(farm.handTools, key)
                            break
                        end
                    end
                end
            end

            if self.isServer then
                currentUser:removeDeleteListener(self, "onPlayerDelete")
            end
        end
    end

    if self.isClient then
        self:updateVisibilityNodes(equipmentInUse)
    end
end

function HandPlantSaplings:onPlayerFarmChanged(player)
    if self.equipmentInUse and self:getIsActivePlayer(player) then
        self:setEquipmentUsed(false, player, false)
    end
end

function HandPlantSaplings:onPlayerDelete(player)
    if self.equipmentInUse and self:getIsActivePlayer(player) then
        self:setEquipmentUsed(false, player, false)
    end
end

function HandPlantSaplings:updateBackpack(fillLevel, treeType, noEventSend)
    HPS_UpdateBackpackEvent.sendEvent(self, fillLevel, treeType, noEventSend)

    if fillLevel > 0 then
        self.backpack.treeType = treeType
    else
        self.backpack.treeType = 0
    end

    self.backpack.fillLevel = fillLevel

    if self.isClient then
        local visNodes = self.visibilityNodes.saplingsBackpack
        if visNodes ~= nil and visNodes.nodes ~= nil then
            for i = 1, #visNodes.nodes do
                setVisibility(visNodes.nodes[i], i <= fillLevel)
            end
        end
    end
end

function HandPlantSaplings:updateStorage(fillLevel, treeType, noEventSend)
    HPS_UpdateStorageEvent.sendEvent(self, fillLevel, treeType, noEventSend)

    if fillLevel > 0 then
        self.storage.treeType = treeType
    else
        self.storage.treeType = 0
    end

    self.storage.fillLevel = fillLevel

    if self.isClient then
        local visNodes = self.visibilityNodes.saplingsStorage
        if visNodes ~= nil and visNodes.nodes ~= nil then
            for i = 1, #visNodes.nodes do
                setVisibility(visNodes.nodes[i], i <= fillLevel)
            end
        end
    end
end

function HandPlantSaplings:updateVisibilityNodes(equipmentInUse)
    if self.isClient then
        if self.visibilityNodes.equipment ~= nil then
            setVisibility(self.visibilityNodes.equipment, not equipmentInUse)
        end

        local backpackSpallingsVisNode = self.visibilityNodes.saplingsBackpack
        if backpackSpallingsVisNode ~= nil and backpackSpallingsVisNode.groupNode ~= nil then
            setVisibility(backpackSpallingsVisNode.groupNode, not equipmentInUse)
        end
    end
end

function HandPlantSaplings:getSaplingTitleFromType(treeTypeIndex)
    if treeTypeIndex ~= nil and treeTypeIndex > 0 then
        local desc = g_treePlantManager:getTreeTypeDescFromIndex(treeTypeIndex)
        if desc ~= nil then
            if self.i18n:hasText(desc.nameI18N) then
                return self.i18n:getText(desc.nameI18N)
            else
                return desc.name
            end
        end
    end

    return "N/A"
end

function HandPlantSaplings:getValidTreeTypeIndex(treeTypeIndex, key)
    if treeTypeIndex > 0 and g_treePlantManager:getTreeTypeDescFromIndex(treeTypeIndex) == nil then
        if key ~= nil then
            print(string.format("Warning: [HandPlantSaplings.lua] Unknown treeTypeIndex '%s' found at %s. Loading default treeTypeIndex '1' instead.", treeTypeIndex, key))
        end

        return 1
    end

    return treeTypeIndex
end

function HandPlantSaplings:getBackpackLevel()
    return self.backpack.treeType, self.backpack.fillLevel
end

function HandPlantSaplings:getStorageLevel()
    return self.storage.treeType, self.storage.fillLevel
end

function HandPlantSaplings:getBackpackSpace()
    return self.backpack.capacity - self.backpack.fillLevel
end

function HandPlantSaplings:getStorageSpace()
    return self.storage.capacity - self.storage.fillLevel
end

function HandPlantSaplings:getCanInteract(player)
    if g_currentMission.controlPlayer and self.playerInRange then
        if player.activeSaplingEquipment ~= nil then
            return player.activeSaplingEquipment == self
        end

        return true
    end

    return false
end

function HandPlantSaplings:getEquipmentIsActive(player)
    if player ~= nil and player:hasHandtoolEquipped() then
        return player.baseInformation.currentHandtool.configFileName == self.handtoolFileName
    end

    return false
end

function HandPlantSaplings:getIsActivePlayer(player)
    if self.currentUser ~= nil then
        return self.currentUser == player
    end

    return false
end

function HandPlantSaplings:getPlayerName(player)
    if player ~= nil then
        return player.visualInformation.playerName
    end

    return "N/A"
end

function HandPlantSaplings:updateActionEvents(register, force)
    if (self.actionEventActive ~= register) or force then
        self.actionEventActive = register

        if g_currentMission.missionDynamicInfo.isMultiplayer then
            if register then
                local isValidUser = self.isServer or g_currentMission.isMasterUser
                if not isValidUser then
                    local farm = g_farmManager:getFarmById(g_currentMission:getFarmId())
                    isValidUser = farm:isUserFarmManager(g_currentMission.playerUserId)
                end

                if isValidUser then
                    if not self.equipmentInUse or (self.equipmentInUse and self:getIsActivePlayer(g_currentMission.player)) then
                        g_inputBinding:beginActionEventsModification(Player.INPUT_CONTEXT_NAME)

                        local _, eventId = g_inputBinding:registerActionEvent(InputAction.TOGGLE_BEACON_LIGHTS, self, self.onMultiplayerLocked, false, true, false, true)
                        g_inputBinding:setActionEventText(eventId, self.i18n:getText("ui_permissions"))
                        self.multiplayerLockEventId = eventId

                        g_inputBinding:endActionEventsModification()
                    end
                end
            else
                if self.multiplayerLockEventId ~= nil then
                    g_inputBinding:beginActionEventsModification(Player.INPUT_CONTEXT_NAME)

                    g_inputBinding:removeActionEventsByTarget(self)
                    self.multiplayerLockEventId = nil

                    g_inputBinding:endActionEventsModification()
                end
            end
        end
    end
end

function HandPlantSaplings:isPermittedUser()
    if g_currentMission.missionDynamicInfo.isMultiplayer and self.adminUsersOnly then
        if self.isServer or g_currentMission.isMasterUser then
            return true
        end

        local farm = g_farmManager:getFarmById(g_currentMission:getFarmId())
        return farm:isUserFarmManager(g_currentMission.playerUserId)
    end

    return true
end

function HandPlantSaplings:onMultiplayerLocked(_, inputValue)
    local text = ""
    if self.adminUsersOnly then
        text = self.i18n:getText("hps_mpUsersAdmin")
    else
        text = self.i18n:getText("hps_mpUsersAll")
    end

    g_gui:showYesNoDialog({text = text, title = self.i18n:getText("ui_permissions"), callback = self.onClickYes, target = self})
end

function HandPlantSaplings:onClickYes(yes)
    if yes then
        self.adminUsersOnly = not self.adminUsersOnly
        if self.isServer then
            g_server:broadcastEvent(HPS_SetUserEvent:new(self, self.adminUsersOnly))
        else
            g_client:getServerConnection():sendEvent(HPS_SetUserEvent:new(self, self.adminUsersOnly))
        end
    end
end

function HandPlantSaplings:interactionTriggerCallback(triggerId, otherId, onEnter, onLeave, onStay, otherShapeId)
    if onEnter or onLeave then
        if g_currentMission.player ~= nil and otherId == g_currentMission.player.rootNode then
            if g_currentMission.accessHandler:canFarmAccessOtherId(g_currentMission:getFarmId(), self:getOwnerFarmId()) then
                if onEnter then
                    if not self.playerInRange then
                        self.playerInRange = true

                        self.activateText = self.i18n:getText("ui_details")
                        g_currentMission:addActivatableObject(self)
                    end
                else
                    if self.playerInRange then
                        self.playerInRange = false

                        g_currentMission:removeActivatableObject(self)
                        self:updateActionEvents(false, true)
                    end
                end

                self:raiseActive()
            end
        end
    end
end

function HandPlantSaplings:onSell() -- Check Type
    HandPlantSaplings:superClass().onSell(self)

    if self.isServer and self.totalFillTypeSellPrice > 0 then
        g_currentMission:addMoney(self.totalFillTypeSellPrice, self:getOwnerFarmId(), MoneyType.PURCHASE_SAPLINGS, true, true)
        self.totalFillTypeSellPrice = 0
    end
end

function HandPlantSaplings:canBeSold()
    local warning = ""

    if self.currentUser ~= nil then
        local controllerName = self.currentUserName or "N/A"
        warning = string.format(self.i18n:getText("hps_sellWarningOne"), controllerName)

        return false, warning
    end

    warning = self.i18n:getText("hps_sellWarningTwo")
    local totalFillLevel = 0
    self.totalFillTypeSellPrice = 0

    if (self.storage.fillLevel > 0) or (self.backpack.fillLevel > 0) then
        totalFillLevel = self.backpack.fillLevel + self.storage.fillLevel

        local price = totalFillLevel * 0.80
        warning = string.format("%s (%d %s) - %s: %s\n", warning, totalFillLevel, self.i18n:getText("unit_pieces"), self.i18n:getText("ui_sellValue"), self.i18n:formatMoney(price, 0, true, true))
        self.totalFillTypeSellPrice = self.totalFillTypeSellPrice + price
    end

    if totalFillLevel > 0 then
        return true, warning
    end

    return true, nil
end

function HandPlantSaplings:onActivateObject()
    local player = g_currentMission.player

    if player ~= nil then
        if self:isPermittedUser() then
            if self.equipmentInUse then
                if player.activeSaplingEquipment ~= nil then
                    if self:getIsActivePlayer(player) then
                        self:setEquipmentUsed(false, player, false)
                    else
                        local warningText = string.format(self.i18n:getText("hps_wrongStorage"), self:getPlayerName(player))
                        g_currentMission:showBlinkingWarning(warningText, HandPlantSaplings.WARNING_TIME)
                    end
                else
                    local controllerName = self.currentUserName or "N/A"
                    g_currentMission:showBlinkingWarning(string.format(self.i18n:getText("hps_inUseByOther"), controllerName), HandPlantSaplings.WARNING_TIME)
                end
            else
                if player.activeSaplingEquipment == nil then
                    self:setEquipmentUsed(true, player, false)
                else
                    local warningText = string.format(self.i18n:getText("hps_inUseByUser"), self:getPlayerName(player))
                    g_currentMission:showBlinkingWarning(warningText, HandPlantSaplings.WARNING_TIME)
                end
            end
        else
            g_gui:showInfoDialog({visible = true, text = self.i18n:getText("shop_messageNoPermissionGeneral"), dialogType = DialogElement.TYPE_INFO, isCloseAllowed = true})
        end
    end
end

function HandPlantSaplings:drawActivate()
    return
end

function HandPlantSaplings:getIsActivatable()
    local controlPlayer = g_currentMission.controlPlayer
    self:updateActionEvents(controlPlayer, not controlPlayer)

    return controlPlayer
end

function HandPlantSaplings:shouldRemoveActivatable()
    return false
end
