--[[
Interface: 1.4.1.0 b5334

Copyright (C) GtX (Andy), 2018

Author: GtX | Andy
Date: 09.12.2018
Version: 1.0.0.0

Contact:
https://forum.giants-software.com
https://github.com/GtX-Andy

History:
V 0.1.0.0 @ 09.12.2018 - Internal Beta
V 0.1.1.0 @ 17.02.2019 - Test Version
V 1.0.0.0 @ 02.05.2019 - Release Version
V 1.1.0.0 @ 17.07.2019 - Add support to position 'waterAddon' parts on unsupported mods.
V 1.1.0.1 @ 07.09.2019 - Fix for Animal Pens built into maps in a weird way.

Important:
Not to be added to any mods / maps or modified from its current release form.
No changes are to be made to this script without permission from GtX | Andy

Darf nicht zu Mods / Maps hinzugefügt oder von der aktuellen Release-Form geändert werden.
An diesem Skript dürfen ohne Genehmigung von GtX | Andy keine Änderungen vorgenommen werden
]]


AnimalPenExtension = {}

local AnimalPenExtension_mt = Class(AnimalPenExtension, Object)
InitObjectClass(AnimalPenExtension, "AnimalPenExtension")

-- Set real world times of service.
-- 12hrs is fair. People have a life you know. ;-)
AnimalPenExtension.SALES_OPEN_TIME = 6
AnimalPenExtension.SALES_CLOSED_TIME = 18

AnimalPenExtension.BUILD_WAITING = -2
AnimalPenExtension.BUILD_START = -1
AnimalPenExtension.BUILD_STARTED = 0

AnimalPenExtension.UNIVERSAL_PARTS_VERSION = 3

g_animalPenExtensionManager.animalPenExtension = AnimalPenExtension

function AnimalPenExtension:new(isServer, isClient, customMt)
    return Object:new(isServer, isClient, customMt or AnimalPenExtension_mt)
end

function AnimalPenExtension:load(owner, xmlFile, key, isMod, hasUniversalParts, extensionBaseDirectory, texts)
    self.owner = owner
    self.nodeId = owner.nodeId
    self.waterLinkNode = owner.nodeId

    self.baseDirectory = owner.baseDirectory
    self.customEnvironment = owner.customEnvironment

    self.hasUniversalParts = hasUniversalParts
    self.universalPartsSet = false
    self.waterPlacementActive = false

    self.partsBaseDirectory = extensionBaseDirectory

    if self.hasUniversalParts then
        self.partsI3dFilename = "sharedParts/universalParts.i3d"
    else
        self.partsI3dFilename = "sharedParts/animalPenParts.i3d"
    end

    self.brand = "FARMING INNOVATIONS"

    self.isMod = isMod
    self.texts = texts

    self.animalTypeTitle = ""
    local animalType = self.owner:getAnimalType()
    if animalType ~= nil then
        local animals = g_animalManager:getAnimalsByType(animalType)
        if animals ~= nil and animals.subTypes ~= nil then
            local fillTypeDesc = animals.subTypes[1].fillTypeDesc
            self.animalTypeTitle = fillTypeDesc.title
        end
    end

    if self.isMod then
        -- Allow mods to use custom water and milk parts.
        local partsI3dFilename = getXMLString(xmlFile, key .. "#partsI3DFilename")
        if partsI3dFilename ~= nil then
            if fileExists(self.baseDirectory .. partsI3dFilename) then
                self.partsBaseDirectory = self.baseDirectory
                self.partsI3dFilename = partsI3dFilename

                -- Only allow 'custom brand name' if user is using custom parts i3d.
                local customBrand = getXMLString(xmlFile, key .. "#customBrandName")
                if customBrand ~= nil and customBrand ~= "" then
                    self.brand = customBrand
                end
            else
                g_animalPenExtensionManager:logPrint(2, "Water Addon parts using i3D filename '%s' could not be loaded! Using 'Base Extension' parts at (%s)", partsI3dFilename, key)
            end
        end
    end

    if g_animalPenExtensionManager.chickenAddonActive and animalType == "CHICKEN" then
        local waterTroughPosition = StringUtil.getVectorNFromString(getXMLString(xmlFile, key .. ".chickenAddon.waterTrough#position"), 3)
        if waterTroughPosition ~= nil then
            local i3dNode = g_i3DManager:loadSharedI3DFile(self.partsI3dFilename, self.partsBaseDirectory, false, false)
            if i3dNode ~= 0 then
                local waterTroughRotation = StringUtil.getRadiansFromString(Utils.getNoNil(getXMLString(xmlFile, key .. ".chickenAddon.waterTrough#rotation"), "0 0 0"), 3)
                local waterTroughNode = I3DUtil.indexToObject(i3dNode, getXMLString(xmlFile, key .. ".chickenAddon.waterTrough#sharedI3dNode"))
                if waterTroughNode ~= nil then
                    local newModule = HusbandryModuleBase.createModule("water")
                    if newModule ~= nil then
                        link(self.nodeId, waterTroughNode)

                        self.chickenAddon = {waterTroughNode = waterTroughNode}
                        setTranslation(self.chickenAddon.waterTroughNode, waterTroughPosition[1], waterTroughPosition[2], waterTroughPosition[3])
                        setRotation(self.chickenAddon.waterTroughNode, waterTroughRotation[1], waterTroughRotation[2], waterTroughRotation[3])

                        local waterTroughScale = StringUtil.getVectorNFromString(getXMLString(xmlFile, key .. ".chickenAddon.waterTrough#scale"), 3)
                        if waterTroughScale ~= nil then
                            setScale(self.chickenAddon.waterTroughNode, waterTroughScale[1], waterTroughScale[2], waterTroughScale[3])
                        end

                        local nodeIndex = getChildIndex(self.chickenAddon.waterTroughNode)
                        if nodeIndex ~= nil then
                            local memoryXML = '<config node="' .. tostring(nodeIndex) .. '|0" exactFillRootNode="0" fillTypes="WATER" acceptedToolTypes="undefined trailer">/n' ..
                                '    <fillPlane node="' .. tostring(nodeIndex) .. '|1" minY="0.06" maxY="0.135" colorChange="true" />/n</config>'

                            local memoryXmlFile = loadXMLFileFromMemory("MemoryXML", memoryXML)
                            if newModule:load(memoryXmlFile, "config", self.nodeId, self.owner, self.baseDirectory) then
                                local levelArea = {}
                                levelArea.start = getChildAt(waterTroughNode, 2)
                                if levelArea.start ~= nil then
                                    levelArea.width = getChildAt(levelArea.start, 0)
                                    levelArea.height = getChildAt(levelArea.start, 1)
                                    if levelArea.width ~= nil and levelArea.height ~= nil then
                                        levelArea.groundType = Utils.getNoNil(getXMLString(xmlFile, key .. ".chickenAddon.waterTrough#triggerAreaGroundType"), "grass")
                                        table.insert(self.owner.levelAreas, levelArea)
                                    end
                                end

                                local triggerMarker = getChildAt(waterTroughNode, 3)
                                if triggerMarker ~= nil then
                                    self.chickenAddon.triggerMarker = triggerMarker
                                    g_currentMission:addTriggerMarker(triggerMarker)
                                end

                                self.owner.modulesByName["water"] = newModule
                                newModule.moduleName = "water"
                                table.insert(self.owner.modulesById, newModule)
                            else
                                self.chickenAddon = nil
                            end

                            delete(memoryXmlFile)
                            memoryXmlFile = nil
                            memoryXML = nil
                        end
                    end
                end

                delete(i3dNode)
                i3dNode = nil
            end
        end
    end

    if g_animalPenExtensionManager.waterAddonActive == true and self.owner.modulesByName["water"] ~= nil then
        if self.hasUniversalParts then
            local waterModule = self.owner.modulesByName["water"]
            if waterModule.unloadPlace ~= nil and waterModule.unloadPlace.exactFillRootNode ~= nil then
                -- Fix for map makers that build placeable pens into the map at World Position 0, 0, 0.
                -- Second fix, create new TG and set location so scale issues are resolved on some maps.
                self.waterLinkNode = createTransformGroup("waterLinkNode")
                link(self.nodeId, self.waterLinkNode)

                local wlnX, wlnY, wlnZ = localToLocal(waterModule.unloadPlace.exactFillRootNode, self.nodeId, 0, 0, 0)
                local dx,dy,dz = localDirectionToLocal(waterModule.unloadPlace.exactFillRootNode, self.nodeId, 0, 0, 1)
                local upx,upy,upz = localDirectionToLocal(waterModule.unloadPlace.exactFillRootNode, self.nodeId, 0, 1, 0)

                setDirection(self.waterLinkNode, dx,dy,dz, upx,upy,upz)
                setTranslation(self.waterLinkNode, wlnX, wlnY, wlnZ)
            end
        end

        local triggerPosition = StringUtil.getVectorNFromString(getXMLString(xmlFile, key .. ".waterAddon.trigger#position"), 3)
        if triggerPosition ~= nil then
            local i3dNode = g_i3DManager:loadSharedI3DFile(self.partsI3dFilename, self.partsBaseDirectory, false, false)
            if i3dNode ~= 0 then
                local triggerNode = I3DUtil.indexToObject(i3dNode, getXMLString(xmlFile, key .. ".waterAddon.trigger#sharedI3dNode"))
                if triggerNode ~= nil then
                    self.waterAddon = {}
                    self.waterAddon.parts = {}
                    self.waterAddon.triggerId = triggerNode

                    self.waterAddon.isActive = false
                    self.waterAddon.displayInfo = true

                    self.waterAddon.rainWater = 0
                    self.waterAddon.updateMinute = 0
                    self.waterAddon.handleSoundActive = false

                    self.waterAddon.decoIsActive = false
                    self.waterAddon.hasBeenPurchased = false
                    self.waterAddon.buildHour = AnimalPenExtension.BUILD_WAITING

                    self.waterAddon.fillLitersPerSecond = Utils.getNoNil(getXMLInt(xmlFile, key .. ".waterAddon#fillLitersPerSecond"), 450)

                    -- If you want a more realistic feel then you can set a priceScale.
                    -- Water Cost is calculated using: Game water cost * waterPriceScale.
                    if g_animalPenExtensionManager.waterPriceScaleOverride < 0 then
                        self.waterAddon.waterPriceScale = Utils.getNoNil(getXMLFloat(xmlFile, key .. ".waterAddon#waterPriceScale"), 0)
                    else
                        self.waterAddon.waterPriceScale = g_animalPenExtensionManager.waterPriceScaleOverride
                    end

                    -- The daily cost to own the Water Addon. Good option if you do not want a water cost.
                    if g_animalPenExtensionManager.maintenancePerDayOverride < 0 then
                        self.waterAddon.maintenanceCost = Utils.getNoNil(getXMLInt(xmlFile, key .. ".waterAddon#maintenancePerDay"), 0)
                    else
                        self.waterAddon.maintenanceCost = g_animalPenExtensionManager.maintenancePerDayOverride
                    end

                    -- Set '0' for indoor troughs.
                    self.waterAddon.rainInputPerHour = Utils.getNoNil(getXMLInt(xmlFile, key .. ".waterAddon#rainInputPerHour"), 60)

                    -- This is the amount it cost to construct the waterAddon.
                    self.waterAddon.buildCost = Utils.getNoNil(getXMLInt(xmlFile, key .. ".waterAddon.construction#buildCost"), 5800)

                    local buildTotalHours = Utils.getNoNil(getXMLInt(xmlFile, key .. ".waterAddon.construction#buildTotalHours"), 10)
                    self.waterAddon.buildTotalHours = math.max(buildTotalHours, 1)

                    if g_animalPenExtensionManager.purchaseOverride then
                        self.waterAddon.requiresPurchase = false
                    else
                        self.waterAddon.requiresPurchase = Utils.getNoNil(getXMLBool(xmlFile, key .. ".waterAddon.construction#requiresPurchase"), false)
                    end

                    local triggerMarker = I3DUtil.indexToObject(i3dNode, getXMLString(xmlFile, key .. ".waterAddon.trigger.marker.sharedI3dNode"))
                    local triggerRotation = StringUtil.getRadiansFromString(Utils.getNoNil(getXMLString(xmlFile, key .. ".waterAddon.trigger#rotation"), "0 0 0"), 3)
                    setTranslation(self.waterAddon.triggerId, triggerPosition[1], triggerPosition[2], triggerPosition[3])
                    setRotation(self.waterAddon.triggerId, triggerRotation[1], triggerRotation[2], triggerRotation[3])
                    addTrigger(self.waterAddon.triggerId, "waterAddonTriggerCallback", self)
                    link(self.waterLinkNode, self.waterAddon.triggerId)

                    -- Adds a 'ground marker' that is connected to the Giants 'SETTINGS MENU' and is a user option
                    if triggerMarker ~= nil then
                        self.waterAddon.triggerMarker = triggerMarker

                        -- If no position or rotation is given then we use the trigger position or rotation as needed.
                        local markerPosition = Utils.getNoNil(StringUtil.getVectorNFromString(getXMLString(xmlFile, key .. ".waterAddon.trigger.marker#position"), 3), triggerPosition)
                        local markerRotation = Utils.getNoNil(StringUtil.getRadiansFromString(getXMLString(xmlFile, key .. ".waterAddon.trigger.marker#rotation"), 3), triggerRotation)
                        setTranslation(self.waterAddon.triggerMarker, markerPosition[1], markerPosition[2], markerPosition[3])
                        setRotation(self.waterAddon.triggerMarker, markerRotation[1], markerRotation[2], markerRotation[3])

                        link(self.waterLinkNode, triggerMarker)
                        g_currentMission:addTriggerMarker(triggerMarker)
                    end

                    local activatable = WaterAddonActivatable:new(self)
                    if activatable ~= nil then
                        self.waterAddon.activatable = activatable
                    else
                        self.waterAddon = nil
                    end
                end

                delete(i3dNode)
                i3dNode = nil
            else
                g_animalPenExtensionManager:logPrint(3, "Failed to load i3d! %s%s  (%s)", self.partsI3dFilename, self.partsBaseDirectory, self.customEnvironment)
            end

            if self.waterAddon ~= nil then
                local valveOperationAdded = false  -- You can only have a single valve operation.

                local i = 0
                while true do
                    local partsKey = string.format("%s.waterAddon.parts.part(%d)", key, i)
                    if not hasXMLProperty(xmlFile, partsKey) then
                        break
                    end

                    local part = {}
                    local node = getXMLString(xmlFile, partsKey .. "#sharedI3dNode")
                    if node ~= nil then
                        part.position = StringUtil.getVectorNFromString(getXMLString(xmlFile, partsKey .. "#position"), 3)
                        if part.position ~= nil then
                            local i3dNode = g_i3DManager:loadSharedI3DFile(self.partsI3dFilename, self.partsBaseDirectory, false, false)
                            if i3dNode ~= 0 then
                                part.node = I3DUtil.indexToObject(i3dNode, node)
                                if part.node ~= nil then
                                    part.rotation = StringUtil.getRadiansFromString(Utils.getNoNil(getXMLString(xmlFile, partsKey .. "#rotation"), "0 0 0"), 3)

                                    if self.isClient then
                                        local isSpout = Utils.getNoNil(getXMLBool(xmlFile, partsKey .. "#isSpout"), false)
                                        if isSpout then
                                            local effectNode = I3DUtil.indexToObject(part.node, getUserAttribute(part.node, "effectNode"))
                                            if effectNode ~= nil then
                                                local effects = g_effectManager:loadEffect(xmlFile, partsKey .. ".effects", effectNode, self)
                                                if effects ~= nil then
                                                    g_effectManager:setFillType(effects, FillType.WATER)

                                                    if self.waterAddon.effects == nil then
                                                        self.waterAddon.effects = {}
                                                    end

                                                    table.insert(self.waterAddon.effects, effects)
                                                end
                                            end

                                            local soundNode = I3DUtil.indexToObject(part.node, getUserAttribute(part.node, "soundNode"))
                                            if soundNode ~= nil then
                                                local sample = g_soundManager:loadSampleFromXML(xmlFile, partsKey .. ".sounds", "waterFilling", self.baseDirectory, soundNode, 0, AudioGroup.ENVIRONMENT, nil, nil)
                                                if sample ~= nil then
                                                    if self.waterAddon.samplesWater == nil then
                                                        self.waterAddon.samplesWater = {}
                                                    end

                                                    table.insert(self.waterAddon.samplesWater, sample)
                                                end
                                            end
                                        else
                                            if not valveOperationAdded then
                                                local isValve = Utils.getNoNil(getXMLBool(xmlFile, partsKey .. "#isValve"), false)
                                                if isValve then
                                                    valveOperationAdded = true

                                                    local valveNode = I3DUtil.indexToObject(part.node, getUserAttribute(part.node, "valveHandleNode"))
                                                    local valveStemNode = I3DUtil.indexToObject(part.node, getUserAttribute(part.node, "valveStemNode"))
                                                    if valveNode ~= nil then
                                                        self.waterAddon.sampleHandle = g_soundManager:loadSampleFromXML(xmlFile, partsKey .. ".sounds", "valveMoving", self.baseDirectory, valveNode, 0, AudioGroup.ENVIRONMENT, nil, nil)

                                                        if hasXMLProperty(xmlFile, partsKey .. ".animation") then
                                                            local clipName = getXMLString(xmlFile, partsKey .. ".animation#clipName")
                                                            if clipName ~= nil then
                                                                local animCharSet = getAnimCharacterSet(valveNode)
                                                                local clip = getAnimClipIndex(animCharSet, clipName)
                                                                if clip ~= nil then
                                                                    local animDuration = getAnimClipDuration(animCharSet, clip);
                                                                    self.waterAddon.valveAnimationClip = {name = clipName, characterSet = animCharSet, duration = animDuration, speedScale = 0}
                                                                    assignAnimTrackClip(animCharSet, 0, clip)
                                                                    setAnimTrackLoopState(animCharSet, 0, false)
                                                                    setAnimTrackSpeedScale(animCharSet, 0, 1)
                                                                end
                                                            else
                                                                if hasXMLProperty(xmlFile, partsKey .. ".animation.valveHandle") then
                                                                    -- For custom valve animation if you do not have a animation clip in custom parts i3d
                                                                    local duration = Utils.getNoNil(getXMLFloat(xmlFile, partsKey .. ".animation#duration"), 2) * 1000
                                                                    self.waterAddon.valveAnimation = {duration = duration, animTime = 0, direction = 1, parts = {}}

                                                                    local startTrans = getXMLString(xmlFile, partsKey .. ".animation.valveHandle#startTrans")
                                                                    local endTrans = getXMLString(xmlFile, partsKey .. ".animation.valveHandle#endTrans")
                                                                    local startRot = getXMLString(xmlFile, partsKey .. ".animation.valveHandle#startRot")
                                                                    local endRot = getXMLString(xmlFile, partsKey .. ".animation.valveHandle#endRot")
                                                                    self:addValveAnimationPart(valveNode, startTrans, endTrans, startRot, endRot)

                                                                    if valveStemNode ~= nil then
                                                                        startTrans = getXMLString(xmlFile, partsKey .. ".animation.valveStem#startTrans")
                                                                        endTrans = getXMLString(xmlFile, partsKey .. ".animation.valveStem#endTrans")
                                                                        startRot = getXMLString(xmlFile, partsKey .. ".animation.valveStem#startRot")
                                                                        endRot = getXMLString(xmlFile, partsKey .. ".animation.valveStem#endRot")
                                                                        self:addValveAnimationPart(valveStemNode, startTrans, endTrans, startRot, endRot)
                                                                    end
                                                                end
                                                            end
                                                        end
                                                    end
                                                end
                                            end
                                        end

                                        part.preBuildPosition = StringUtil.getVectorNFromString(getXMLString(xmlFile, partsKey .. "#preBuildPosition"), 3)
                                        if part.preBuildPosition ~= nil then
                                            part.preBuildRotation = StringUtil.getRadiansFromString(Utils.getNoNil(getXMLString(xmlFile, partsKey .. "#preBuildRotation"), "0 0 0"), 3)
                                        end

                                        part.collisionNode = I3DUtil.indexToObject(part.node, getUserAttribute(part.node, "collisionNode"))
                                        if part.collisionNode ~= nil then
                                            part.collisionNodeRigidBody = getRigidBodyType(part.collisionNode)
                                        end

                                        -- Scale is only set on load if parts need changes.
                                        local scale = StringUtil.getVectorNFromString(getXMLString(xmlFile, partsKey .. "#scale"), 3)
                                        if scale ~= nil then
                                            setScale(part.node, scale[1], scale[2], scale[3])
                                        end

                                        if self.waterAddon.requiresPurchase then
                                            if part.preBuildPosition ~= nil then
                                                setTranslation(part.node, part.preBuildPosition[1], part.preBuildPosition[2], part.preBuildPosition[3])
                                                setRotation(part.node, part.preBuildRotation[1], part.preBuildRotation[2], part.preBuildRotation[3])
                                            else
                                                setTranslation(part.node, part.position[1], part.position[2], part.position[3])
                                                setRotation(part.node, part.rotation[1], part.rotation[2], part.rotation[3])
                                            end

                                            part.rigidBody = getRigidBodyType(part.node)
                                            setRigidBodyType(part.node, "NoRigidBody")
                                            setVisibility(part.node, false)

                                            if part.collisionNode ~= nil then
                                                setRigidBodyType(part.collisionNode, "NoRigidBody")
                                            end
                                        else
                                            setTranslation(part.node, part.position[1], part.position[2], part.position[3])
                                            setRotation(part.node, part.rotation[1], part.rotation[2], part.rotation[3])
                                            setVisibility(part.node, true)
                                        end
                                    end

                                    part.isPartDelay = false

                                    link(self.waterLinkNode, part.node)
                                    table.insert(self.waterAddon.parts, part)
                                end

                                delete(i3dNode)
                                i3dNode = nil
                            else
                                g_animalPenExtensionManager:logPrint(3, "Failed to load i3d! %s%s  (%s)", self.partsI3dFilename, self.partsBaseDirectory, self.customEnvironment)
                            end
                        end
                    else
                        -- Allow fake build parts to be added to extend the time between displayed items.
                        -- This way you can show an item at a specific build time if wanted.
                        if Utils.getNoNil(getXMLBool(xmlFile, partsKey .. "#isPartDelay"), false) then
                            local delayValue = Utils.getNoNil(getXMLInt(xmlFile, partsKey .. "#delayValue"), 1)
                            if delayValue > 1 then
                                for i = 1, delayValue do
                                    table.insert(self.waterAddon.parts, {isPartDelay = true})
                                end
                            else
                                table.insert(self.waterAddon.parts, {isPartDelay = true})
                            end
                        end
                    end

                    i = i + 1
                end

                if self.waterAddon.requiresPurchase then
                    local purchaseMarkerPosition = StringUtil.getVectorNFromString(getXMLString(xmlFile, key .. ".waterAddon.construction.marker#position"), 3)
                    if purchaseMarkerPosition ~= nil then
                        local i3dNode = g_i3DManager:loadSharedI3DFile(self.partsI3dFilename, self.partsBaseDirectory, false, false)
                        if i3dNode ~= 0 then
                            local purchaseMarkerNode = I3DUtil.indexToObject(i3dNode, getXMLString(xmlFile, key .. ".waterAddon.construction.marker#sharedI3dNode"))
                            if purchaseMarkerNode ~= nil then
                                local purchaseMarkerRotation = StringUtil.getRadiansFromString(Utils.getNoNil(getXMLString(xmlFile, key .. ".waterAddon.construction.marker#rotation"), "0 0 0"), 3)

                                setTranslation(purchaseMarkerNode, purchaseMarkerPosition[1], purchaseMarkerPosition[2], purchaseMarkerPosition[3])
                                setRotation(purchaseMarkerNode, purchaseMarkerRotation[1], purchaseMarkerRotation[2], purchaseMarkerRotation[3])

                                local removeOn = getXMLString(xmlFile, key .. ".waterAddon.construction.marker#removeOn")
                                if removeOn == nil then
                                    removeOn = "START"
                                end

                                local removeType = removeOn:upper()
                                local removeStage = AnimalPenExtension.BUILD_STARTED
                                if removeType == "PURCHASE" then
                                    removeStage = AnimalPenExtension.BUILD_START
                                elseif removeType == "FINISH" then
                                    removeStage = self.waterAddon.buildTotalHours
                                end

                                setVisibility(purchaseMarkerNode, true)
                                link(self.nodeId, purchaseMarkerNode)

                                self.waterAddon.purchaseMarker = {node = purchaseMarkerNode, removeStage = removeStage}
                            end

                            delete(i3dNode)
                            i3dNode = nil
                        end
                    end

                    i = 0
                    while true do
                        local decorationPartKey = string.format("%s.waterAddon.construction.decorationParts.part(%d)", key, i)
                        if not hasXMLProperty(xmlFile, decorationPartKey) then
                            break
                        end

                        local extraPart = {}
                        extraPart.isActive = false
                        extraPart.position = StringUtil.getVectorNFromString(getXMLString(xmlFile, decorationPartKey .. "#position"), 3)
                        if extraPart.position ~= nil then
                            extraPart.rotation = StringUtil.getRadiansFromString(Utils.getNoNil(getXMLString(xmlFile, decorationPartKey .. "#rotation"), "0 0 0"), 3)

                            extraPart.scale = StringUtil.getVectorNFromString(getXMLString(xmlFile, decorationPartKey .. "#scale"), 3)

                            local filename = getXMLString(xmlFile, decorationPartKey .. "#filename")
                            if filename ~= nil then
                                -- Allow access to 'animalPenParts.i3d' file also if you wish to load parts for deco from this.
                                if filename:sub(1, 19) == "$animalPenParts.i3d" then
                                    extraPart.filename = self.partsI3dFilename
                                    extraPart.baseDirectory = extensionBaseDirectory
                                else
                                    if fileExists(self.baseDirectory .. filename) then
                                        extraPart.filename = filename
                                        extraPart.baseDirectory = self.baseDirectory
                                    else
                                        g_animalPenExtensionManager:logPrint(2, "Water Addon parts using i3D filename '%s%s' could not be loaded! (%s)", self.baseDirectory, filename, decorationPartKey)
                                    end
                                end
                            else
                                extraPart.filename = "sharedParts/decoParts.i3d"
                                extraPart.baseDirectory = extensionBaseDirectory
                            end

                            if extraPart.filename ~= nil then
                                extraPart.sharedI3dNode = Utils.getNoNil(getXMLString(xmlFile, decorationPartKey .. "#sharedI3dNode"), "0")

                                if self.waterAddon.deco == nil then
                                    self.waterAddon.deco = {}
                                end

                                table.insert(self.waterAddon.deco, extraPart)
                            end
                        end

                        i = i + 1
                    end
                else
                    self.waterAddon.buildHour = self.waterAddon.buildTotalHours
                end
            end
        end
    end

    local milkModule = self.owner.modulesByName["milk"]
    if g_animalPenExtensionManager.milkAddonActive == true and milkModule ~= nil then
        local triggerPosition = StringUtil.getVectorNFromString(getXMLString(xmlFile, key .. ".milkAddon.trigger#position"), 3)
        if triggerPosition ~= nil then
            local i3dNode = g_i3DManager:loadSharedI3DFile(self.partsI3dFilename, self.partsBaseDirectory, false, false)
            if i3dNode ~= 0 then
                local triggerNode = I3DUtil.indexToObject(i3dNode, getXMLString(xmlFile, key .. ".milkAddon.trigger#sharedI3dNode"))
                if triggerNode ~= nil then
                    self.milkAddon = {}
                    self.milkAddon.triggerId = triggerNode

                    local triggerMarker = I3DUtil.indexToObject(i3dNode, getXMLString(xmlFile, key .. ".milkAddon.trigger.marker#sharedI3dNode"))
                    local triggerRotation = StringUtil.getRadiansFromString(Utils.getNoNil(getXMLString(xmlFile, key .. ".milkAddon.trigger#rotation"), "0 0 0"), 3)
                    setTranslation(self.milkAddon.triggerId, triggerPosition[1], triggerPosition[2], triggerPosition[3])
                    setRotation(self.milkAddon.triggerId, triggerRotation[1], triggerRotation[2], triggerRotation[3])
                    addTrigger(self.milkAddon.triggerId, "milkAddonTriggerCallback", self)
                    link(self.nodeId, self.milkAddon.triggerId)

                    -- Adds a 'ground marker' that is connected to the Giants 'SETTINGS MENU' and is a user option
                    if triggerMarker ~= nil then
                        self.milkAddon.triggerMarker = triggerMarker

                        -- If no position or rotation is given then we use the trigger position or rotation as needed.
                        local markerPosition = Utils.getNoNil(StringUtil.getVectorNFromString(getXMLString(xmlFile, key .. ".milkAddon.trigger.marker#position"), 3), triggerPosition)
                        local markerRotation = Utils.getNoNil(StringUtil.getRadiansFromString(getXMLString(xmlFile, key .. ".milkAddon.trigger.marker#rotation"), 3), triggerRotation)
                        setTranslation(self.milkAddon.triggerMarker, markerPosition[1], markerPosition[2], markerPosition[3])
                        setRotation(self.milkAddon.triggerMarker, markerRotation[1], markerRotation[2], markerRotation[3])

                        link(self.nodeId, triggerMarker)
                        g_currentMission:addTriggerMarker(triggerMarker)
                    end

                    local activatable = MilkAddonActivatable:new(self)
                    if activatable ~= nil then
                        self.milkAddon.activatable = activatable
                        canLoad = true
                    else
                        self.milkAddon = nil
                    end
                end

                delete(i3dNode)
                i3dNode = nil
            else
                g_animalPenExtensionManager:logPrint(3, "Failed to load i3d! %s%s  (%s)", self.partsI3dFilename, self.partsBaseDirectory, self.customEnvironment)
            end

            if self.milkAddon ~= nil then
                local salesAreaPosition = StringUtil.getVectorNFromString(Utils.getNoNil(getXMLString(xmlFile, key .. ".milkAddon.salesArea#position"), "0 0 0"), 3)
                if salesAreaPosition ~= nil then
                    local i3dNode = g_i3DManager:loadSharedI3DFile(self.partsI3dFilename, self.partsBaseDirectory, false, false)
                    if i3dNode ~= 0 then
                        local salesAreaNode = I3DUtil.indexToObject(i3dNode, getXMLString(xmlFile, key .. ".milkAddon.salesArea#sharedI3dNode"))
                        if salesAreaNode ~= nil then
                            local salesAreaRotation = StringUtil.getRadiansFromString(getXMLString(xmlFile, key .. ".milkAddon.salesArea#rotation"), 3)

                            setTranslation(salesAreaNode, salesAreaPosition[1], salesAreaPosition[2], salesAreaPosition[3])
                            setRotation(salesAreaNode, salesAreaRotation[1], salesAreaRotation[2], salesAreaRotation[3])

                            if Utils.getNoNil(getXMLBool(xmlFile, key .. ".milkAddon.salesArea.display#useSalesAreaNode"), false) then
                                self.milkAddon.display = salesAreaNode
                            else
                                self.milkAddon.display = I3DUtil.indexToObject(salesAreaNode, getXMLString(xmlFile, key .. ".milkAddon.salesArea.display#node"))
                            end

                            self.milkAddon.salesOpenSign = I3DUtil.indexToObject(salesAreaNode, getXMLString(xmlFile, key .. ".milkAddon.salesArea.signs#openNode"))
                            self.milkAddon.salesClosedSign = I3DUtil.indexToObject(salesAreaNode, getXMLString(xmlFile, key .. ".milkAddon.salesArea.signs#closedNode"))

                            link(self.nodeId, salesAreaNode)

                            if self.milkAddon.display ~= nil then
                                I3DUtil.setNumberShaderByValue(self.milkAddon.display, 0, 0, true)
                                milkModule.setFillLevel = Utils.appendedFunction(milkModule.setFillLevel, self.setMilkFillLevel)
                            end
                        end

                        delete(i3dNode)
                        i3dNode = nil
                    else
                        g_animalPenExtensionManager:logPrint(3, "Failed to load i3d! %s%s  (%s)", self.partsI3dFilename, self.partsBaseDirectory, self.customEnvironment)
                    end
                end

                -- Allow custom mods to add extra delay texts if they want.
                if g_animalPenExtensionManager.randomDelayTexts ~= nil then
                    local i = 0
                    while true do
                        local randomTextKey = string.format("%s.milkAddon.randomDelayTexts.text(%d)", key, i)
                        if not hasXMLProperty(xmlFile, randomTextKey) then
                            break
                        end

                        local name = getXMLString(xmlFile, randomTextKey .. "#name")
                        if name ~= nil and g_i18n:hasText(name) then
                            if g_animalPenExtensionManager.randomDelayTextsUsed[name] == nil then
                                g_animalPenExtensionManager.randomDelayTextsUsed[name] = true
                                local text = g_i18n:getText(name)
                                table.insert(g_animalPenExtensionManager.randomDelayTexts, text)
                            end
                        end

                        i = i + 1
                    end
                end

                self:setMilkSaleSigns((not self:getRandomDelayActive() and self:getIsSalesOpen()))
            end
        end
    end

    --if hasXMLProperty(xmlFile, key .. ".grassGrazing") then
        -- My Unreleased placeable version from FS17. Was not released as 'RM' released there version before I finished.
        -- Removed after RM announced they will add this type of feature in Seasons. Again :rofl
        -- May add later if I do not like their version or for people who do not use Seasons.
    --end

    if g_currentMission ~= nil and g_currentMission.environment ~= nil then
        if self.isServer and self.waterAddon ~= nil and self.waterAddon.maintenanceCost > 0 then
            g_currentMission.environment:addDayChangeListener(self)
        end

        if self.waterAddon ~= nil or self.milkAddon ~= nil then
            g_currentMission.environment:addHourChangeListener(self)
        end
    end

    return (self.waterAddon ~= nil) or (self.milkAddon ~= nil) or (self.chickenAddon ~= nil)
end

function AnimalPenExtension:addValveAnimationPart(node, startTrans, endTrans, startRot, endRot)
    local animCurve = AnimCurve:new(linearInterpolatorN)
    local sx, sy, sz = StringUtil.getVectorFromString(startTrans)
    local ex, ey, ez = StringUtil.getVectorFromString(endTrans)
    local srx, sry, srz = StringUtil.getVectorFromString(startRot)
    local erx, ery, erz = StringUtil.getVectorFromString(endRot)
    local nx,ny,nz = getTranslation(node)
    local nrx,nry,nrz = getRotation(node)

    srx = Utils.getNoNilRad(srx, nrx)
    sry = Utils.getNoNilRad(sry, nry)
    srz = Utils.getNoNilRad(srz, nrz)
    sx = Utils.getNoNil(sx, nx)
    sy = Utils.getNoNil(sy, ny)
    sz = Utils.getNoNil(sz, nz)
    animCurve:addKeyframe({sx, sy, sz, srx, sry, srz, time = 0})

    erx = Utils.getNoNilRad(erx, nrx)
    ery = Utils.getNoNilRad(ery, nry)
    erz = Utils.getNoNilRad(erz, nrz)
    ex = Utils.getNoNil(ex, nx)
    ey = Utils.getNoNil(ey, ny)
    ez = Utils.getNoNil(ez, nz)
    animCurve:addKeyframe({ex, ey, ez, erx, ery, erz, time = 1})

    setTranslation(node, sx, sy, sz)
    setRotation(node, srx, sry, srz)

    table.insert(self.waterAddon.valveAnimation.parts, {node = node, animCurve = animCurve})
end

function AnimalPenExtension:delete()
    if self.chickenAddon ~= nil and self.chickenAddon.triggerMarker ~= nil then
        g_currentMission:removeTriggerMarker(self.chickenAddon.triggerMarker)
    end

    if self.waterAddon ~= nil then
        if self.waterPlacementActive then
            g_animalPenExtensionManager.universalPlacement:cancelOrDelete(false, false)
        end

        if self.waterAddon.triggerId ~= nil then
            if self.waterAddon.playerInRange ~= nil then
                g_currentMission:removeActivatableObject(self.waterAddon.activatable)
                self.waterAddon.playerInRange = nil
            end

            removeTrigger(self.waterAddon.triggerId)
            self.waterAddon.triggerId = nil

            if self.waterAddon.triggerMarker ~= nil then
                g_currentMission:removeTriggerMarker(self.waterAddon.triggerMarker)
            end

            if self.partsBaseDirectory ~= nil and self.partsI3dFilename ~= nil then
                g_i3DManager:releaseSharedI3DFile(self.partsI3dFilename, self.partsBaseDirectory, true)
            end
        end

        if self.isClient then
            if self.waterAddon.effects ~= nil then
                for id, effect in pairs(self.waterAddon.effects) do
                    g_effectManager:deleteEffects(effect)
                end

                self.waterAddon.effects = nil
            end

            if self.waterAddon.samplesWater ~= nil then
                for _, sample in pairs (self.waterAddon.samplesWater) do
                    g_soundManager:deleteSample(self.waterAddon.samplesWater)
                end

                self.waterAddon.samplesWater = nil
            end

            if self.waterAddon.sampleHandle ~= nil then
                g_soundManager:deleteSample(self.waterAddon.sampleHandle)
                self.waterAddon.sampleHandle = nil
            end
        end

        if self.waterAddon.deco ~= nil then
            for _, extraPart in pairs(self.waterAddon.deco) do
                if extraPart.node ~= nil and extraPart.filename ~= nil and extraPart.baseDirectory ~= nil then
                    delete(extraPart.node)
                    extraPart.node = nil

                    g_i3DManager:releaseSharedI3DFile(extraPart.filename, extraPart.baseDirectory, true)
                end
            end
        end

        if self.waterAddon.parts ~= nil then
            if self.partsBaseDirectory ~= nil and self.partsI3dFilename ~= nil then
                for _, part in pairs(self.waterAddon.parts) do
                    if part.node ~= nil then
                        g_i3DManager:releaseSharedI3DFile(self.partsI3dFilename, self.partsBaseDirectory, true)
                    end
                end
            end
        end

        self.waterAddon = nil
    end

    if self.milkAddon ~= nil then
        if self.milkAddon.triggerId ~= nil then
            if self.milkAddon.playerInRange ~= nil then
                g_currentMission:removeActivatableObject(self.milkAddon.activatable)
                self.milkAddon.playerInRange = nil
            end

            removeTrigger(self.milkAddon.triggerId)
            self.milkAddon.triggerId = nil

            if self.milkAddon.triggerMarker ~= nil then
                g_currentMission:removeTriggerMarker(self.milkAddon.triggerMarker)
            end

            if self.partsBaseDirectory ~= nil and self.partsI3dFilename ~= nil then
                if self.milkAddon.display ~= nil then
                    g_i3DManager:releaseSharedI3DFile(self.partsI3dFilename, self.partsBaseDirectory, true)
                end
            end
        end

        self.milkAddon = nil
    end

    if g_currentMission ~= nil and g_currentMission.environment ~= nil then
        if self.isServer then
            g_currentMission.environment:removeDayChangeListener(self)
        end

        g_currentMission.environment:removeHourChangeListener(self)
    end

    self.partsBaseDirectory = nil
    self.partsI3dFilename = nil
end

function AnimalPenExtension:loadFromXMLFile(xmlFile, key)
    if self.waterAddon ~= nil then
        if self.waterAddon.requiresPurchase then
            local buildHour = getXMLInt(xmlFile, key .. ".waterAddon#buildHour")
            if buildHour ~= nil and buildHour >= AnimalPenExtension.BUILD_WAITING then
                self:setWaterAddonBuildHour(buildHour)
            end

            if self.waterAddon.buildHour >= AnimalPenExtension.BUILD_START then
                self.waterAddon.hasBeenPurchased = true
            else
                self.waterAddon.hasBeenPurchased = Utils.getNoNil(getXMLBool(xmlFile, key .. ".waterAddon#hasBeenPurchased"), false)
            end
        end

        if self.hasUniversalParts then
            local vX, vY, vZ = StringUtil.getVectorFromString(getXMLString(xmlFile, key .. ".universalParts#valvePosition"))
            local vRY = tonumber(getXMLString(xmlFile, key .. ".universalParts#valveRotation"))

            if vX ~= nil and vY ~= nil and vZ ~= nil and vRY ~= nil then
                local version = getXMLInt(xmlFile, key .. ".universalParts#version")
                if version ~= nil and version == AnimalPenExtension.UNIVERSAL_PARTS_VERSION then
                    local sX, sY, sZ = StringUtil.getVectorFromString(getXMLString(xmlFile, key .. ".universalParts#spoutPosition"))
                    local sRY = tonumber(getXMLString(xmlFile, key .. ".universalParts#spoutRotation"))

                    if sX ~= nil and sY ~= nil and sZ ~= nil and sRY ~= nil then
                        local vRX, _, vRZ = getRotation(self.waterAddon.parts[1].node)
                        vRY = math.rad(vRY)

                        local sRX, _, sRZ = getRotation(self.waterAddon.parts[2].node)
                        sRY = math.rad(sRY)

                        self:setUniversalParts(vX, vY, vZ, vRY, sX, sY, sZ, sRY)
                    end
                else
                    local name = self.customEnvironment
                    if name == nil then
                        name = self.baseDirectory
                    end
                    g_animalPenExtensionManager:logPrint(1, "Something went wrong or 'Universal Parts' version has changed! Resetting valve and spout locations to avoid conflicts for mod '%s'.", name)
                end
            end

        end
    end

    return true
end

function AnimalPenExtension:saveToXMLFile(xmlFile, key, usedModNames)
    if self.waterAddon ~= nil then
        if self.waterAddon.requiresPurchase then
            setXMLInt(xmlFile, key .. ".waterAddon#buildHour", self.waterAddon.buildHour)
            setXMLBool(xmlFile, key .. ".waterAddon#hasBeenPurchased", self.waterAddon.hasBeenPurchased)
        end

        if self.hasUniversalParts and self.universalPartsSet then
            setXMLInt(xmlFile, key .. ".universalParts#version", AnimalPenExtension.UNIVERSAL_PARTS_VERSION)

            local vX, vY, vZ, _, vRY, _, sX, sY, sZ, _, sRY, _ = self:getUniversalParts()

            setXMLString(xmlFile, key .. ".universalParts#valvePosition", string.format("%.4f %.4f %.4f", vX, vY, vZ))
            setXMLString(xmlFile, key .. ".universalParts#valveRotation", string.format("%.4f", math.deg(vRY)))

            setXMLString(xmlFile, key .. ".universalParts#spoutPosition", string.format("%.4f %.4f %.4f", sX, sY, sZ))
            setXMLString(xmlFile, key .. ".universalParts#spoutRotation", string.format("%.4f", math.deg(sRY)))
        end
    end
end

function AnimalPenExtension:readStream(streamId, connection)
    local animalPenExtensionId = NetworkUtil.readNodeObjectId(streamId)

    if self.waterAddon ~= nil then
        local isActive = streamReadBool(streamId)
        self:setFillingState(isActive, true)

        local buildHour = streamReadInt8(streamId)
        self:setWaterAddonBuildHour(buildHour)
        self.waterAddon.hasBeenPurchased = buildHour >= AnimalPenExtension.BUILD_START

        if streamReadBool(streamId) then
            local vX = streamReadFloat32(streamId)
            local vY = streamReadFloat32(streamId)
            local vZ = streamReadFloat32(streamId)
            local vRY = NetworkUtil.readCompressedAngle(streamId)

            local sX = streamReadFloat32(streamId)
            local sY = streamReadFloat32(streamId)
            local sZ = streamReadFloat32(streamId)
            local sRY = NetworkUtil.readCompressedAngle(streamId)

            self:setUniversalParts(vX, vY, vZ, vRY, sX, sY, sZ, sRY)
        end
    end

    g_client:finishRegisterObject(self, animalPenExtensionId)
end

function AnimalPenExtension:writeStream(streamId, connection)
    NetworkUtil.writeNodeObjectId(streamId, NetworkUtil.getObjectId(self))

    if self.waterAddon ~= nil then
        streamWriteBool(streamId, self.waterAddon.isActive)
        streamWriteInt8(streamId, self.waterAddon.buildHour)

        if streamWriteBool(streamId, self.universalPartsSet) then
            local vX, vY, vZ, _, vRY, _, sX, sY, sZ, _, sRY, _ = self:getUniversalParts()

            streamWriteFloat32(streamId, vX)
            streamWriteFloat32(streamId, vY)
            streamWriteFloat32(streamId, vZ)
            NetworkUtil.writeCompressedAngle(streamId, vRY)

            streamWriteFloat32(streamId, sX)
            streamWriteFloat32(streamId, sY)
            streamWriteFloat32(streamId, sZ)
            NetworkUtil.writeCompressedAngle(streamId, sRY)
        end
    end

    g_server:registerObjectInStream(connection, self)
end

function AnimalPenExtension:hourChanged()
    if self.isClient and self.milkAddon ~= nil then
        local state = not self:getRandomDelayActive() and self:getIsSalesOpen()
        if state ~= self.milkAddon.salesSignState then
            self:setMilkSaleSigns(state)
        end
    end

    if self.isServer and self.waterAddon ~= nil then
        if self.waterAddon.rainInputPerHour > 0 and g_currentMission.environment.weather:getIsRaining() then
            self:raiseActive()
        end

        if self.waterAddon.requiresPurchase and self.waterAddon.buildHour < self.waterAddon.buildTotalHours then
            local currentHour = g_currentMission.environment.currentHour
            if currentHour >= AnimalPenExtension.SALES_OPEN_TIME and currentHour < AnimalPenExtension.SALES_CLOSED_TIME then
                if self.waterAddon.buildHour >= AnimalPenExtension.BUILD_STARTED then
                    self:setWaterAddonBuildHour(self.waterAddon.buildHour + 1)
                else
                    if self.waterAddon.buildHour == AnimalPenExtension.BUILD_START then
                        if currentHour == AnimalPenExtension.SALES_OPEN_TIME then
                            self:setWaterAddonBuildHour(AnimalPenExtension.BUILD_STARTED)
                        end
                    end
                end
            end
        end
    end
end

function AnimalPenExtension:setWaterAddonBuildHour(buildHour)
    self.waterAddon.buildHour = buildHour

    if self.waterAddon.requiresPurchase then
        if g_server ~= nil then
            g_server:broadcastEvent(AnimalPenExtensionBuildHourEvent:new(self, buildHour))

            if buildHour == AnimalPenExtension.BUILD_START and not self.waterAddon.hasBeenPurchased then
                self.waterAddon.hasBeenPurchased = true
                g_currentMission:addMoney(-self.waterAddon.buildCost, self:getOwnerFarmId(), MoneyType.PROPERTY_MAINTENANCE, true, true)
            end
        end

        if self.waterAddon.deco ~= nil and self.waterAddon.buildHour >= AnimalPenExtension.BUILD_STARTED then
            if self.waterAddon.buildHour < self.waterAddon.buildTotalHours then
                if not self.waterAddon.decoIsActive then
                    self.waterAddon.decoIsActive = true

                    for _, extraPart in pairs(self.waterAddon.deco) do
                        if extraPart.node == nil and extraPart.filename ~= nil and extraPart.baseDirectory ~= nil then
                            local i3dNode = g_i3DManager:loadSharedI3DFile(extraPart.filename, extraPart.baseDirectory, false, false)
                            if i3dNode ~= 0 then
                                extraPart.node = I3DUtil.indexToObject(i3dNode, extraPart.sharedI3dNode)

                                setTranslation(extraPart.node, extraPart.position[1], extraPart.position[2], extraPart.position[3])
                                setRotation(extraPart.node, extraPart.rotation[1], extraPart.rotation[2], extraPart.rotation[3])

                                if extraPart.scale ~= nil then
                                    setScale(extraPart.node, extraPart.scale[1], extraPart.scale[2], extraPart.scale[3])
                                end

                                link(self.nodeId, extraPart.node)
                                addToPhysics(extraPart.node)

                                delete(i3dNode)
                            end
                        end
                    end
                end
            else
                for _, extraPart in pairs(self.waterAddon.deco) do
                    if extraPart.node ~= nil and extraPart.filename ~= nil and extraPart.baseDirectory ~= nil then
                        delete(extraPart.node)
                        extraPart.node = nil
                        g_i3DManager:releaseSharedI3DFile(extraPart.filename, extraPart.baseDirectory, true)
                    end
                end

                self.waterAddon.deco = nil
            end
        end

        if self.waterAddon.parts ~= nil then
            local numberOfParts = #self.waterAddon.parts
            local maxNode = math.ceil((numberOfParts * self.waterAddon.buildHour) / self.waterAddon.buildTotalHours)
            for i = 1, numberOfParts do
                local part = self.waterAddon.parts[i]

                if part.node ~= nil then
                    if i <= maxNode then
                        if part.preBuildPosition ~= nil then
                            setVisibility(part.node, buildHour >= AnimalPenExtension.BUILD_STARTED)

                            if not part.posRotSet then
                                setTranslation(part.node, part.position[1], part.position[2], part.position[3])
                                setRotation(part.node, part.rotation[1], part.rotation[2], part.rotation[3])
                                setRigidBodyType(part.node, part.rigidBody or "NoRigidBody")

                                if part.collisionNode ~= nil then
                                    setRigidBodyType(part.collisionNode, part.collisionNodeRigidBody or "NoRigidBody")
                                end

                                part.posRotSet = true
                            end
                        else
                            setVisibility(part.node, true)
                            setRigidBodyType(part.node, part.rigidBody or "NoRigidBody")

                            if part.collisionNode ~= nil then
                                setRigidBodyType(part.collisionNode, part.collisionNodeRigidBody or "NoRigidBody")
                            end
                        end
                    else
                        if part.preBuildPosition ~= nil then
                            setVisibility(part.node, buildHour >= AnimalPenExtension.BUILD_STARTED)

                            if part.posRotSet == nil then
                                if part.collisionNode ~= nil then
                                    setRigidBodyType(part.collisionNode, "NoRigidBody")
                                end

                                setRigidBodyType(part.node, "NoRigidBody")
                                setTranslation(part.node, part.preBuildPosition[1], part.preBuildPosition[2], part.preBuildPosition[3])
                                setRotation(part.node, part.preBuildRotation[1], part.preBuildRotation[2], part.preBuildRotation[3])

                                part.posRotSet = false
                            end
                        else
                            setVisibility(part.node, false)
                            setRigidBodyType(part.node, "NoRigidBody")

                            if part.collisionNode ~= nil then
                                setRigidBodyType(part.collisionNode, "NoRigidBody")
                            end
                        end
                    end
                end
            end
        end

        if self.waterAddon.purchaseMarker ~= nil then
            if buildHour >= self.waterAddon.purchaseMarker.removeStage then
                setVisibility(self.waterAddon.purchaseMarker.node, false)
                delete(self.waterAddon.purchaseMarker.node)
                self.waterAddon.purchaseMarker = nil
            else
                setVisibility(self.waterAddon.purchaseMarker.node, true)
            end
        end
    end
end

function AnimalPenExtension:getIsWaterAddonOwned()
    if self.hasUniversalParts then
        return self.universalPartsSet
    end

    return self.waterAddon.buildHour >= self.waterAddon.buildTotalHours
end

function AnimalPenExtension:getIsSalesOpen(currentHour)
    local currentHour = g_currentMission.environment.currentHour
    return currentHour >= AnimalPenExtension.SALES_OPEN_TIME and currentHour < AnimalPenExtension.SALES_CLOSED_TIME
end

function AnimalPenExtension:dayChanged()
    if self.isServer then
        if self.waterAddon ~= nil and self.waterAddon.maintenanceCost > 0 then
            local _, capacity = self:getWaterLevels()
            if capacity > 0 then
                local cost = self.waterAddon.maintenanceCost
                g_currentMission:addMoney(-cost, self:getOwnerFarmId(), MoneyType.PROPERTY_MAINTENANCE, true)
            end
        end
    end
end

function AnimalPenExtension:update(dt)
    if self.waterAddon ~= nil then
        local playerInTrigger = self.waterAddon.playerInRange ~= nil and self.waterAddon.playerInRange == g_currentMission.player
        if playerInTrigger then
            if self:getIsWaterAddonOwned() then
                if self.owner:getNumOfAnimals() > 0 then
                    local fillLevel, capacity = self:getWaterLevels()
                    local info = self.texts.waterFillLevel .. ":  " .. math.floor(fillLevel) .. " / " .. math.floor(capacity) .. "  (" .. math.floor(100 * fillLevel / capacity) .. "%)"
                    g_currentMission:addExtraPrintText(info)

                    -- if g_animalPenExtensionManager.renderUI then
                        -- Future 'RenderUI' update.
                    -- end
                else
                    g_currentMission:addExtraPrintText(self.texts.noAnimalsWarning)

                    -- if g_animalPenExtensionManager.renderUI then
                        -- Future 'RenderUI' update.
                    -- end
                end
            end

            self:raiseActive()
        end

        if self.waterAddon.isActive then
            if self.isServer then
                local waterModule = self.owner.modulesByName["water"]
                local amountToAdd = waterModule:getFreeCapacity(FillType.WATER)
                local fillDelta = math.min(amountToAdd, self.waterAddon.fillLitersPerSecond * dt * 0.001)

                self:updateTrough(waterModule, fillDelta, true)
            end

            self:raiseActive()

            self:fillEffectState(true)
        else
            self:fillEffectState(false)
        end

        if self.isClient then
            if self.waterAddon.valveAnimationClip ~= nil then
                if self.waterAddon.isActive or self.waterAddon.valveAnimationClip.speedScale ~= 0 then
                    local currentTime = getAnimTrackTime(self.waterAddon.valveAnimationClip.characterSet, 0)
                    local isClipActive = false

                    if self.waterAddon.valveAnimationClip.speedScale == 1 then
                        isClipActive = currentTime < self.waterAddon.valveAnimationClip.duration
                    elseif self.waterAddon.valveAnimationClip.speedScale == -1 then
                        isClipActive = currentTime > 0.0
                    end

                    if isClipActive then
                        self:raiseActive()
                    else
                        disableAnimTrack(self.waterAddon.valveAnimationClip.characterSet, 0)
                        self.waterAddon.valveAnimationClip.speedScale = 0
                    end

                    self:updateValveSound(isClipActive)
                end
            elseif self.waterAddon.valveAnimation ~= nil then
                if self.waterAddon.isActive or (self.waterAddon.valveAnimation.animTime > 0) or self.waterAddon.handleSoundActive then
                    self:updateValveAnimation(self.waterAddon.isActive, dt)
                end
            end
        end

        if self.isServer and self.waterAddon.rainInputPerHour > 0 then
            if g_currentMission.environment.weather:getIsRaining() then
                self.waterAddon.updateMinute = self.waterAddon.updateMinute + (dt * g_currentMission.loadingScreen.missionInfo.timeScale)
                if self.waterAddon.updateMinute >= 60000 then
                    self.waterAddon.updateMinute = 0

                    local rainToAdd = (self.waterAddon.rainInputPerHour / 60) * g_currentMission.environment.weather:getRainFallScale()
                    self.waterAddon.rainWater = self.waterAddon.rainWater + rainToAdd

                    if self.waterAddon.rainWater >= 15 then
                        self:updateTrough(self.owner.modulesByName["water"], self.waterAddon.rainWater)
                        self.waterAddon.rainWater = 0
                    end
                end

                self:raiseActive()
            else
                if self.waterAddon.rainWater ~= 0 then
                    if self.waterAddon.rainWater > 7 then
                        self:updateTrough(self.owner.modulesByName["water"], self.waterAddon.rainWater)
                    end

                    self.waterAddon.rainWater = 0

                    if self.waterAddon.updateMinute ~= 0 then
                        self.waterAddon.updateMinute = 0
                    end
                end
            end

        end
    end
end

function AnimalPenExtension:setFillingState(state, noEventSend)
    AnimalPenExtensionFillEvent.sendEvent(self, state, noEventSend)
    self.waterAddon.isActive = state

    if self.isClient then
        if self.waterAddon.valveAnimationClip ~= nil then
            local animTime, speedScale = 0.0, 1
            enableAnimTrack(self.waterAddon.valveAnimationClip.characterSet, 0)
            local currentTime = getAnimTrackTime(self.waterAddon.valveAnimationClip.characterSet, 0)

            if state then
                animTime = math.max(0.0, currentTime)
            else
                speedScale = -1
                animTime = math.min(currentTime, self.waterAddon.valveAnimationClip.duration)
            end

            setAnimTrackTime(self.waterAddon.valveAnimationClip.characterSet, 0, animTime)
            setAnimTrackSpeedScale(self.waterAddon.valveAnimationClip.characterSet, 0, speedScale)
            self.waterAddon.valveAnimationClip.speedScale = speedScale
        end
    end

    self:raiseActive()
end

function AnimalPenExtension:fillEffectState(isActive)
    if self.isClient then
        if self.waterAddon.effectsActive ~= isActive then
            if self.waterAddon.effects ~= nil then
                for _, effect in pairs (self.waterAddon.effects) do
                    if isActive then
                        g_effectManager:setFillType(effect, FillType.WATER)
                        g_effectManager:startEffects(effect)
                    else
                        g_effectManager:stopEffects(effect)
                    end
                end
            end

            if self.waterAddon.samplesWater ~= nil then
                for _, sample in pairs (self.waterAddon.samplesWater) do
                    if isActive then
                        g_soundManager:playSample(sample)
                    else
                        g_soundManager:stopSample(sample)
                    end
                end
            end
        end
        self.waterAddon.effectsActive = isActive
    end
end

function AnimalPenExtension:updateValveAnimation(positive, dt)
    local valveAnimation = self.waterAddon.valveAnimation

    local direction = -1
    if positive then
        direction = 1
    end

    local doUpdate = true
    if direction > 0 then
        if valveAnimation.animTime == 1 then
            doUpdate = false
        end
    else
        if valveAnimation.animTime == 0 then
            doUpdate = false
        end
    end

    if doUpdate then
        valveAnimation.animTime = MathUtil.clamp(valveAnimation.animTime + direction * dt / valveAnimation.duration, 0, 1)

        for _, part in pairs (valveAnimation.parts) do
            local v = part.animCurve:get(valveAnimation.animTime)
            setTranslation(part.node, v[1], v[2], v[3])
            setRotation(part.node, v[4], v[5], v[6])
        end

        self:raiseActive()
    end

    self:updateValveSound(doUpdate)
end

function AnimalPenExtension:updateValveSound(isActive)
    if self.waterAddon.sampleHandle ~= nil then
        if isActive then
            if not self.waterAddon.handleSoundActive then
                self.waterAddon.handleSoundActive = true
                g_soundManager:playSample(self.waterAddon.sampleHandle)
            end
        else
            if self.waterAddon.handleSoundActive then
                self.waterAddon.handleSoundActive = false
                g_soundManager:stopSample(self.waterAddon.sampleHandle)
            end
        end
    end
end

function AnimalPenExtension:updateTrough(waterModule, fillDelta, isTownWater)
    local farmId = self:getOwnerFarmId()

    if fillDelta > 0 and fillDelta <= waterModule:getFreeCapacity(FillType.WATER) then
        waterModule:addFillLevelFromTool(farmId, fillDelta, FillType.WATER)
    else
        if self.waterAddon.isActive then
            self:setFillingState(false)
        end
    end

    if isTownWater == true and self.waterAddon.waterPriceScale > 0 then
        local price = fillDelta * g_currentMission.economyManager:getPricePerLiter(FillType.WATER) * self.waterAddon.waterPriceScale
        g_farmManager:updateFarmStats(farmId, "expenses", price)

        if MoneyType.PURCHASE_WATER ~= nil then
            g_currentMission:addMoney(-price, farmId, MoneyType.PURCHASE_WATER, true)
        else
            g_currentMission:addMoney(-price, farmId, MoneyType.OTHER, true)
        end
    end
end

function AnimalPenExtension:getWaterLevels()
    local fillLevel, capacity = 0, 0

    local waterModule = self.owner.modulesByName["water"]
    if waterModule ~= nil then
        fillLevel = Utils.getNoNil(waterModule.fillLevels[FillType.WATER], 0.0)
        capacity = waterModule.fillCapacity
    end

    return fillLevel, capacity
end

function AnimalPenExtension:getCanWaterActivate()
    if self.owner:getNumOfAnimals() > 0 then
        local waterModule = self.owner.modulesByName["water"]
        if waterModule ~= nil then
            local fillLevel = Utils.getNoNil(waterModule.fillLevels[FillType.WATER], 0.0)
            local capacity = waterModule.fillCapacity

            if capacity > 0 then
                return math.floor(100 * (fillLevel / capacity)) < 98
            end
        end
    end

    return false
end

function AnimalPenExtension:setMilkFillLevel(fillTypeIndex, fillLevel)
    if self.owner.animalPenExtension ~= nil then
        local milkAddon = self.owner.animalPenExtension.milkAddon
        if milkAddon ~= nil and milkAddon.display ~= nil then
            I3DUtil.setNumberShaderByValue(milkAddon.display, fillLevel, 0, true)
        end
    end
end

function AnimalPenExtension:setMilkSaleSigns(state)
    self.milkAddon.salesSignState = state

    if self.milkAddon.salesOpenSign ~= nil then
        setVisibility(self.milkAddon.salesOpenSign, state)
    end

    if self.milkAddon.salesClosedSign ~= nil then
        setVisibility(self.milkAddon.salesClosedSign, not state)
    end
end

function AnimalPenExtension:getRandomDelayActive()
    if g_animalPenExtensionManager ~= nil then
        return g_animalPenExtensionManager.randomDelayActive
    end

    return false
end

function AnimalPenExtension:getMilkLevels()
    local fillLevel, capacity = 0, 0

    local milkModule = self.owner.modulesByName["milk"]
    if milkModule ~= nil then
        fillLevel = Utils.getNoNil(milkModule.fillLevels[FillType.MILK], 0.0)
        capacity = milkModule.fillCapacity
    end

    return fillLevel, capacity
end

function AnimalPenExtension:openMilkContractorMenu()
    if self:getIsSalesOpen() then
        if self:getRandomDelayActive() then
            local text = g_animalPenExtensionManager:getNextRandomText()
            g_gui:showInfoDialog({visible = true, text = text, dialogType = DialogElement.TYPE_INFO, isCloseAllowed = true})
        else
            local dialog = g_gui:showDialog("MilkSaleDialog")
            if dialog ~= nil then
                local fillLevel, capacity = self:getMilkLevels()
                dialog.target:setTitle(self.texts.dialogHeader)
                dialog.target:setData(fillLevel, capacity)
                dialog.target:loadCallback(self.sellMilkCallback, self)
            end
        end
    else
        g_gui:showInfoDialog({visible = true, text = self.texts.salesClosed, dialogType = DialogElement.TYPE_INFO, isCloseAllowed = true})
    end
end

function AnimalPenExtension:sellMilkCallback(amountToSell, contractorFee, sellPoint)
    if amountToSell > 0 and sellPoint ~= nil then
        if g_currentMission:getIsServer() then
            local milkModule = self.owner.modulesByName["milk"]
            local oldMilk = milkModule.fillLevels[FillType.MILK]
            if oldMilk ~= nil and oldMilk >= amountToSell then
                local farmId = self:getOwnerFarmId()
                sellPoint:addFillLevelFromTool(farmId, amountToSell, FillType.MILK)
                local currentMilk = Utils.getNoNil(milkModule.fillLevels[FillType.MILK], 0.0)
                milkModule:setFillLevel(FillType.MILK, currentMilk - amountToSell)
                if contractorFee ~= nil and contractorFee > 0 then
                    g_currentMission:addMoney(-contractorFee, farmId, MoneyType.OTHER, true, true)
                end
            end
        else
            local noNilFee = Utils.getNoNil(contractorFee, 0)
            g_client:getServerConnection():sendEvent(AnimalPenExtensionSellMilkEvent:new(self, amountToSell, noNilFee, sellPoint))
        end
    end
end

function AnimalPenExtension:waterAddonTriggerCallback(triggerId, otherId, onEnter, onLeave, onStay)
    if onEnter or onLeave then
        if g_currentMission.player ~= nil and otherId == g_currentMission.player.rootNode then
            if onEnter then
                if self.ownerFarmId == nil or (self.ownerFarmId == AccessHandler.EVERYONE) or
                    g_currentMission.accessHandler:canFarmAccessOtherId(g_currentMission:getFarmId(), self.ownerFarmId) then

                    if self.waterAddon.playerInRange == nil then
                        g_currentMission:addActivatableObject(self.waterAddon.activatable)
                        self.waterAddon.playerInRange = g_currentMission.player
                    end
                end
            else
                if self.waterAddon.playerInRange ~= nil then
                    g_currentMission:removeActivatableObject(self.waterAddon.activatable)
                    self.waterAddon.playerInRange = nil
                end
            end

            self:raiseActive()
        end
    end
end

function AnimalPenExtension:milkAddonTriggerCallback(triggerId, otherId, onEnter, onLeave, onStay)
    if onEnter or onLeave then
        if g_currentMission.player ~= nil and otherId == g_currentMission.player.rootNode then
            if onEnter then
                if self.ownerFarmId == nil or (self.ownerFarmId == AccessHandler.EVERYONE) or
                    g_currentMission.accessHandler:canFarmAccessOtherId(g_currentMission:getFarmId(), self.ownerFarmId) then

                    if self.milkAddon.playerInRange == nil then
                        g_currentMission:addActivatableObject(self.milkAddon.activatable)
                        self.milkAddon.playerInRange = g_currentMission.player
                    end
                end
            else
                if self.milkAddon.playerInRange ~= nil then
                    g_currentMission:removeActivatableObject(self.milkAddon.activatable)
                    self.milkAddon.playerInRange = nil
                end
            end
        end
    end
end

function AnimalPenExtension:preBuyText(hasPermission)
    -- Adjust 'precision' if the purchase price is less than '0.1' so it does not just show '0'
    local precision = 2
    local cost = g_currentMission.economyManager:getPricePerLiter(FillType.WATER) * self.waterAddon.waterPriceScale
    if cost < 0.1 then
        if cost > 0.01 then
            precision = 3
        else
            precision = 4
        end
    end

    local buildCost = g_i18n:formatMoney(self.waterAddon.buildCost, 0, true, true)
    local waterCost =  string.format("%s: %s / %s.", self.texts.water, g_i18n:formatMoney(cost, precision, true, true), g_i18n:getVolumeUnit(true))
    local maintenanceCost = string.format(g_i18n:getText("shop_maintenanceValue"), g_i18n:formatMoney(self.waterAddon.maintenanceCost, 0, true, true)) .. "."
    local mainText = string.format(self.texts.purchase, buildCost, self.animalTypeTitle)
    local defualtText = string.format("%s\n\n%s %s\n%s", mainText, g_i18n:getText("shop_maintenance"), maintenanceCost, waterCost)

    if hasPermission then
        g_gui:showYesNoDialog({text = defualtText, title = self.brand, callback = self.onClickYes, target = self})
    else
        local text = string.format("%s\n\n%s\n%s", defualtText, self.texts.requiredPermissions, g_i18n:getText("ui_permissions_buyPlaceable"))
        g_gui:showInfoDialog({text = text, okText = g_i18n:getText("button_back")})
    end
end

function AnimalPenExtension:onClickYes(yes)
    if yes then
        if self.isServer then
            self:setWaterAddonBuildHour(AnimalPenExtension.BUILD_START)
            self:showBuyInfo()
        else
            g_client:getServerConnection():sendEvent(AnimalPenExtensionBuildHourEvent:new(self, AnimalPenExtension.BUILD_START))
            self:showBuyInfo()
        end
    end
end

function AnimalPenExtension:showBuyInfo()
    local text, dialogType = "", DialogElement.TYPE_LOADING

    if self.waterAddon.buildHour < AnimalPenExtension.BUILD_STARTED then
        text = string.format(self.texts.bought, self.brand, self.waterAddon.buildTotalHours)
    else
        if self:getIsSalesOpen() then
            text = string.format(self.texts.startedBuild, self.brand, (self.waterAddon.buildTotalHours - self.waterAddon.buildHour))
        else
            dialogType = DialogElement.TYPE_INFO
            text = string.format(self.texts.startedBuildNoPlumber, (self.waterAddon.buildTotalHours - self.waterAddon.buildHour))
        end
    end

    g_gui:showInfoDialog({text = text, dialogType = dialogType})
end

function AnimalPenExtension:getUniversalParts()
    if self.waterAddon.parts ~= nil and #self.waterAddon.parts >= 2 then

        local valve = self.waterAddon.parts[1].node
        local vX, vY, vZ = getTranslation(valve)
        local vRX, vRY, vRZ = getRotation(valve)

        local spout = self.waterAddon.parts[2].node
        local sX, sY, sZ = getTranslation(spout)
        local sRX, sRY, sRZ = getRotation(spout)

        return vX, vY, vZ, vRX, vRY, vRZ, sX, sY, sZ, sRX, sRY, sRZ
    end

    return 0, -0.25, 0, 0, 0, 0, 0, -0.06, 0, 0, 0, 0
end

function AnimalPenExtension:setUniversalParts(vX, vY, vZ, vRY, sX, sY, sZ, sRY)
    if not self.hasUniversalParts or self.universalPartsSet or self.waterAddon == nil then
        return false
    end

    if self.waterAddon.parts ~= nil and #self.waterAddon.parts >= 2 then
        local valve = self.waterAddon.parts[1].node
        removeFromPhysics(valve)

        local vRX, _, vRZ = getRotation(valve)
        setTranslation(valve, vX, vY, vZ)
        setRotation(valve, vRX, vRY, vRZ)
        addToPhysics(valve)

        local trigger = self.waterAddon.triggerId
        removeFromPhysics(trigger)
        setTranslation(trigger, vX, vY, vZ)
        setRotation(trigger, vRX, vRY, vRZ)
        addToPhysics(trigger)

        local spout = self.waterAddon.parts[2].node
        removeFromPhysics(spout)

        local sRX, _, sRZ = getRotation(spout)
        setTranslation(spout, sX, sY, sZ)
        setRotation(spout, sRX, sRY, sRZ)
        addToPhysics(spout)

        self.universalPartsSet = true

        return true
    end

    return false
end

WaterAddonActivatable = {}
local WaterAddonActivatable_mt = Class(WaterAddonActivatable)

function WaterAddonActivatable:new(extension)
    local self = {};
    setmetatable(self, WaterAddonActivatable_mt)

    self.extension = extension
    self.texts = extension.texts
    self.activateText = "TEXT ERROR"

    return self
end

function WaterAddonActivatable:getIsActivatable()
    if self.extension:getIsWaterAddonOwned() then
        if self.extension:getCanWaterActivate() then
            local state = self.extension.waterAddon.isActive
            self:setActivateText(state, true)

            return g_currentMission.controlPlayer and g_currentMission.controlledVehicle == nil
        end
    else
        if self.extension.hasUniversalParts then
            self.activateText = g_i18n:getText("button_configurate")
            return g_currentMission.controlPlayer and g_currentMission.controlledVehicle == nil
        else
            local state = self.extension.waterAddon.buildHour > AnimalPenExtension.BUILD_WAITING
            self:setActivateText(state, false)
            return g_currentMission.controlPlayer and g_currentMission.controlledVehicle == nil
        end
    end

    return false
end

function WaterAddonActivatable:onActivateObject()
    if self.extension:getIsWaterAddonOwned() then
        local newState = not self.extension.waterAddon.isActive
        self.extension:setFillingState(newState)
    else
        if self.extension.hasUniversalParts then
            if g_animalPenExtensionManager.universalPlacement ~= nil then
                if g_currentMission:getHasPlayerPermission("buyPlaceable") then
                    local waterPlacementActive = g_animalPenExtensionManager.universalPlacement:activate(self.extension, self.extension.waterAddon.parts)
                    if waterPlacementActive then
                        self.extension.waterPlacementActive = waterPlacementActive
                        g_currentMission:removeActivatableObject(self)
                        self.extension.waterAddon.playerInRange = nil
                    end
                end
            end
        else
            if self.extension.waterAddon.buildHour > AnimalPenExtension.BUILD_WAITING then
                self.extension:showBuyInfo()
            else
                local hasPermission = g_currentMission:getHasPlayerPermission("buyPlaceable")
                self.extension:preBuyText(hasPermission)
            end
        end
    end
end

function WaterAddonActivatable:shouldRemoveActivatable()
    return false
end

function WaterAddonActivatable:drawActivate()
    return
end

function WaterAddonActivatable:setActivateText(state, isOwned)
    if isOwned then
        if state then
            self.activateText = self.texts.stopFilling
        else
            self.activateText = self.texts.startFilling
        end
    else
        if state then
            self.activateText = self.texts.checkBuildState
        else
            self.activateText = self.texts.doPurchase
        end
    end
end

MilkAddonActivatable = {}
local MilkAddonActivatable_mt = Class(MilkAddonActivatable)

function MilkAddonActivatable:new(extension)
    local self = {};
    setmetatable(self, MilkAddonActivatable_mt)

    self.extension = extension
    if extension.texts.callMilkContractor ~= nil then
        self.activateText = extension.texts.callMilkContractor
    else
        self.activateText = "TEXT ERROR"
    end

    return self
end

function MilkAddonActivatable:getIsActivatable()
    return g_currentMission.controlPlayer and g_currentMission.controlledVehicle == nil
end

function MilkAddonActivatable:onActivateObject()
    self.extension:openMilkContractorMenu()
end

function MilkAddonActivatable:shouldRemoveActivatable()
    return false
end

function MilkAddonActivatable:drawActivate()
    return
end
